#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Cost function for wood chip terminals
# Description: Cost function is split up into wrapper (implemented in pyomo) and core function (actual testable cost function)
#################################################################################################################################################################################################################################################

import pandas as pd
import pyomo.environ as pyo

def implement_wood_chip_shipping_terminal_import_cost_function(model, data):
    """
    :param model: Pyomo model
    :param data: The dataset dict containing node and edge data
    :return: Cost function
    """

    model.nodes_wood_chip_shipping_terminal_import_cost_constraint_07 = pyo.Constraint(model.wc_shipping_terminal_indices_07, rule=lambda model, terminal: cost_function_pyomo(model, terminal, data))


def cost_function_pyomo(model, terminal, data):
    """
    Function that translates the core cost function into pyomo language
    """

    num_incoming_trips = sum(model.edges_wood_chip_shipping_num_trips_06[export_terminal, terminal] for export_terminal in model.wc_shipping_terminal_indices_05) # [#trips per year]
    metadata = data['metadata']

    return model.nodes_wood_chip_shipping_terminal_cost_07[terminal] == cost_function_core(num_incoming_trips, metadata)


def cost_function_core(num_ships_incoming, metadata):
    """
    Cost function for timber import shipping terminals
    :param num_ships_incoming: Number of trips per year [#trips per fleet]
    :param metadata: Metadata dataset
    :return: Total cost # [tNOK2024/a]
    """

    # Port fees
    ship_size_bt = metadata['Timber Ship Gross tonnage [-]']
    ship_deadweight = metadata['Timber Ship Deadweight tonnage [tonnes]']

    quay_fee = 0.598 * ship_deadweight * 11.63 / 1000  # [tNOK2024/Trip]
    time_fee = 0.2716 * ship_size_bt * 11.63 / 1000  # Can increase by 20% if additional days in harbour [tNOK2024/Trip]

    port_fees_import = time_fee + quay_fee  # [tNOK2024/Trip]
    total_cost = port_fees_import * num_ships_incoming

    return total_cost


if __name__ == '__main__':
    print('Testing cost function')

    metadata = pd.read_excel('../data/general/metadata.xlsx', index_col=0)['Value']

    expansion_size_hydrogen_mw = 400
    expansion_size_hydrogen_kta = expansion_size_hydrogen_mw / 100 * 26.3
    expansion_size_timber_kta = expansion_size_hydrogen_kta * metadata['Biomass Gasification OPEX Timber consumption [kg/kgH2]']
    num_trips = expansion_size_timber_kta * 1000 / metadata['Timber Ship Deadweight tonnage [tonnes]']

    total_cost = cost_function_core(num_trips, metadata) # [tNOK2024/a]
    print('Specific production costs: {:.2f} [NOK2024/t timber] -> {:.2f} [NOK2024/kg H2]'.format(total_cost*1000/(expansion_size_timber_kta*1000), total_cost*1000/(expansion_size_hydrogen_kta*1000*1000)))