#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Cost function for wood chip terminals
# Description: Cost function contains cost for wood chip production and port fees
#################################################################################################################################################################################################################################################

import pyomo.environ as pyo
import numpy as np
import pandas as pd

def implement_wood_chip_shipping_terminal_export_cost_function(model, data):
    """
    :param model: Pyomo model
    :param data: The dataset dict containing node and edge data
    :return: Cost function
    """

    model.wood_chip_shipping_terminal_cost_constraint_05 = pyo.Constraint(model.wc_shipping_terminal_indices_05, rule=lambda model, terminal: cost_function_pyomo(model, terminal, data))


def cost_function_pyomo(model, terminal, data):

    expansion_size = model.nodes_wood_chip_shipping_terminal_expansion_size_05[terminal] # [kt timber per year]
    ships_incoming = sum(model.edges_wood_chip_shipping_num_trips_06[terminal, import_terminal] for import_terminal in model.wc_shipping_terminal_indices_07)  # Number of trips per year for all fleets

    return model.nodes_wood_chip_shipping_terminal_cost_05[terminal] == cost_function_core(terminal, ships_incoming, expansion_size, data) # [tNOK2024/a]


def cost_function_core(terminal, num_ships_incoming, expansion_size, data):
    """
    Cost function for shipping terminals for timber export
    :param terminal: The timber shipping terminal (export)
    :param num_ships_incoming: Number of trips per year [#trips per fleet]
    :param expansion_size: Plant size in kt/a timber export
    :param data: The dataset dict containing node and edge data
    :return: Total cost # [tNOK2024/a]
    """

    # Cost for wood chipping
    timber_supply = expansion_size * 1000 * 1000  # [kg/a]
    electricity_price = data['node_data_wood_chip_shipping_terminals_05'].loc[terminal, 'Electricity price [NOK2024/MWh]']
    electricity_consumption_specific = 52  # kJ/kg timber
    electricity_consumption_absolute = electricity_consumption_specific * timber_supply / 3600 / 1000  # [MWh/a]
    electricity_cost = electricity_price * electricity_consumption_absolute / 1000  # [tNOK2024/a]

    annual_costs = electricity_cost  # [tNOK2024/a]

    # Terminal fees
    metadata = data['metadata']
    ship_size_bt = metadata['Timber Ship Gross tonnage [-]']
    ship_deadweight = metadata['Timber Ship Deadweight tonnage [tonnes]']
    days_at_export_terminal = np.ceil(metadata['Timber ship transfer time export terminal [h]'] / 24)  # Fees are paid as per døgn
    node_data_timber_shipping_terminals_05 = data['node_data_wood_chip_shipping_terminals_05']

    if terminal in ['Sortland', 'Tromsø']:
        time_fee = node_data_timber_shipping_terminals_05.loc[terminal, 'Time fees'] * days_at_export_terminal / 1000  # Absolute fee [tNOK2024/Trip]
    elif terminal == 'Larvik':
        time_fee = node_data_timber_shipping_terminals_05.loc[terminal, 'Time fees'] * days_at_export_terminal / 1000 * 215  # Length based fee [tNOK2024/Trip]
    else:
        time_fee = node_data_timber_shipping_terminals_05.loc[terminal, 'Time fees'] * days_at_export_terminal / 1000 * ship_size_bt  # Size based fee [tNOK2024/Trip]

    waterway_fee = node_data_timber_shipping_terminals_05.loc[terminal, 'Waterway fee'] * ship_size_bt / 1000  # [tNOK2024/Trip]
    quay_fee = node_data_timber_shipping_terminals_05.loc[terminal, 'Quay fee wood chips'] * ship_deadweight / 1000  # [tNOK2024/Trip]
    port_fees_export = waterway_fee + time_fee + quay_fee  # [tNOK2024/Trip]
    annual_costs += port_fees_export * num_ships_incoming  # [tNOK2024/a]

    return annual_costs


if __name__ == '__main__':

    print('Testing cost function')

    data_dir = '../data/'
    metadata = pd.read_excel(data_dir + 'general/metadata.xlsx', index_col=0)['Value']

    data = {}
    data['node_data_wood_chip_shipping_terminals_05'] = pd.read_feather(data_dir + 'node_data_05_wc_shipping_terminals_export.feather')  # Index = Timber shipping terminals (export)
    data['metadata'] = metadata

    expansion_size_hydrogen_mw = 200
    expansion_size_hydrogen_kta = expansion_size_hydrogen_mw / 100 * 26.3
    expansion_size_timber_kta = expansion_size_hydrogen_kta * metadata['Biomass Gasification OPEX Timber consumption [kg/kgH2]']

    num_trips = expansion_size_timber_kta * 1000 / metadata['Timber Ship Deadweight tonnage [tonnes]']

    total_cost = cost_function_core('Mandal', num_trips, expansion_size_timber_kta, data)
    print('Specific production costs: {:.2f} [NOK2024/kg H2]'.format(total_cost*1000/(expansion_size_hydrogen_kta*1000*1000)))
