#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Cost function for cH2 shipping terminals
#################################################################################################################################################################################################################################################

import numpy as np
import pyomo.environ as pyo

############################################################################################################################################
# CONSTRAINTS

def implement_ch2_shipping_terminal_import_cost_function(model, metadata):
    """
    :param model: Pyomo model
    :param metadata: The metadata dataset
    """

    model.ch2_shipping_terminal_cost_constraint_07 = pyo.Constraint(model.cH2_shipping_terminal_indices_07, rule=lambda model, terminal: cost_function_pyomo(model, terminal, metadata))


def cost_function_pyomo(model, terminal, metadata):
    """
    Cost function for the cH2 export shipping terminals in pyomo language
    """

    # Variables
    expansion_size_main_plant = model.nodes_ch2_shipping_terminal_expansion_size_07[terminal]  # [kt/a hydrogen]
    expansion_size_main_plant_mw = expansion_size_main_plant / 26.2975 * 100
    expansion_decision_main_plant = model.nodes_ch2_shipping_terminal_expansion_decision_07[terminal]

    # Investment costs
    investment_costs_total = 0.694 * expansion_size_main_plant_mw + 0.24 * metadata['cH2 Ship Num Modules [-]'] + 59.5 # TASC for shipping terminal [MNOK2024]
    investment_costs_total = investment_costs_total * crf(metadata['cH2 Shipping Terminal Lifetime [a]'], metadata['WACC [%]'] / 100) * 1000 # [tNOK2024/a]
    investment_costs_total = investment_costs_total * expansion_decision_main_plant # [tNOK2024/a]

    # Electricity costs
    electricity_price = metadata['Electricity price (DE) [€/MWh]'] * 11.629  # [NOK/MWh]
    electricity_consumption = metadata['cH2 Shipping Terminal (import) OPEX Electricity consumption [kWh/kgH2]'] * expansion_size_main_plant * 1000 * 1000 / 1000  # [MWh/a]
    electricity_costs = electricity_consumption * electricity_price / 1000  # [tNOK2024/a]

    # Other operating costs
    other_opex = 0  # [tNOK2024/a]
    other_opex += metadata['Staff cost - Process operator [tNOK2024/a]'] * 4 * expansion_decision_main_plant # Staff
    other_opex += investment_costs_total / 1.05 / 1.093 * 0.05  # Maintenance, 5% of TPC

    return model.nodes_ch2_shipping_terminal_cost_07[terminal] == investment_costs_total + electricity_costs + other_opex


############################################################################################################################################
# HELPER FUNCTIONS

def crf(lifetime, i):
    return (i*np.power(1+i,lifetime)) / (np.power(1+i,lifetime)-1)


############################################################################################################################################
# TESTING

if __name__ == '__main__':
    pass