#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Cost function for cH2 shipping terminals
#################################################################################################################################################################################################################################################

import numpy as np
import pyomo.environ as pyo

############################################################################################################################################
# CONSTRAINTS

def implement_ch2_shipping_terminal_export_cost_function(model, data):
    """
    :param model: Pyomo model
    :param data: The dataset dict containing node and edge data
    :return: None
    """

    model.ch2_shipping_terminal_cost_constraint_05 = pyo.Constraint(model.cH2_shipping_terminal_indices_05, rule=lambda model, terminal: cost_function_pyomo(model, terminal, data))


def cost_function_pyomo(model, terminal, data):
    """
    Cost function for the cH2 export shipping terminals in pyomo language
    """

    # Parameters
    metadata = data['metadata']
    electricity_price = data['node_data_cH2_shipping_terminals_05'].loc[terminal, 'Electricity price [NOK2024/MWh]']

    # Variables
    expansion_size_main_plant = model.nodes_ch2_shipping_terminal_expansion_size_05[terminal] # [kt/a hydrogen]
    expansion_size_main_plant_mw = expansion_size_main_plant / 26.2975 * 100
    expansion_decision_main_plant = model.nodes_ch2_shipping_terminal_expansion_decision_05[terminal]

    expansion_size_tt_terminal = model.nodes_ch2_shipping_terminal_ch2_tube_trailer_expansion_size_05[terminal] # [kt/a hydrogen]
    expansion_size_tt_terminal_mw = expansion_size_tt_terminal / 26.2975 * 100  # [kt/a H2 -> MW H2]
    expansion_decision_tt_terminal = model.nodes_ch2_shipping_terminal_ch2_tube_trailer_expansion_decision_05[terminal]

    # Investment costs
    investment_costs_with_full_ttt = 1.64 *  expansion_size_main_plant_mw + 197.1 # TASC for shipping terminal which is fully supplied by H2 Tube Trailers [MNOK2024]
    investment_costs_with_full_ttt = investment_costs_with_full_ttt * crf(metadata['cH2 Shipping Terminal Lifetime [a]'], metadata['WACC [%]']/100) # [MNOK2024/a]

    investment_costs_full_ttt = calculate_tube_trailer_terminal_investment(expansion_size_main_plant_mw, metadata) # TASC for the TTT at a shipping terminal which is fully supplied by H2 Tube Trailers [MNOK2024/a]
    investment_costs_without_ttt = investment_costs_with_full_ttt - investment_costs_full_ttt # [MNOK2024/a]
    investment_costs_ttt = calculate_tube_trailer_terminal_investment(expansion_size_tt_terminal_mw, metadata) # TASC for the TTT at the given shipping terminal [MNOK2024/a]

    investment_costs_total = investment_costs_without_ttt * expansion_decision_main_plant + investment_costs_ttt * expansion_decision_tt_terminal # [MNOK2024/a]
    investment_costs_total = investment_costs_total * 1000 # [tNOK2024/a]

    # Electricity cost
    electricity_cost_main_plant = electricity_price / 1000 * metadata['cH2 Shipping Terminal (export) OPEX Electricity consumption Main Plant [kWh/kgH2]'] / 1000  # [tNOK2024/kgH2]
    electricity_cost_main_plant = electricity_cost_main_plant * 1000 * 1000 * expansion_size_main_plant  # [tNOK2024/a]

    electricity_cost_tt_terminal = electricity_price / 1000 * metadata['cH2 Shipping Terminal (export) OPEX Electricity consumption cH2 TT Terminal [kWh/kgH2]'] / 1000  # [tNOK2024/kgH2]
    electricity_cost_tt_terminal = electricity_cost_tt_terminal * 1000 * 1000 * expansion_size_tt_terminal  # [tNOK2024/a]

    electricity_cost = electricity_cost_main_plant + electricity_cost_tt_terminal

    # Other operating costs
    other_opex = 0 # [tNOK2024/a]

    other_opex += metadata['Staff cost - Process operator [tNOK2024/a]'] * 4 * expansion_decision_main_plant # Staff
    other_opex += investment_costs_total / 1.05 / 1.093 * 0.05 # Maintenance, 5% of TPC
    other_opex += expansion_size_main_plant * 1000 * 1000 * 0.056 / 1000 # Cooling water

    return model.nodes_ch2_shipping_terminal_cost_05[terminal] == investment_costs_total + electricity_cost + other_opex

############################################################################################################################################


############################################################################################################################################
# HELPER FUNCTIONS

def calculate_tube_trailer_terminal_investment(tt_terminal_hydrogen_throughput_mw, metadata):
    number_of_receiving_ports = 0.0124*tt_terminal_hydrogen_throughput_mw + 0.505  # [-]
    tt_terminal_TASC = metadata['cH2 Tube Trailer Unloading Terminal TASC [MNOK2024]'] * number_of_receiving_ports # [MNOK2024]
    tt_terminal_TASC = tt_terminal_TASC * crf(metadata['H2 Compressor Lifetime [a]'], metadata['WACC [%]']/100) # [MNOK2024/a]

    return tt_terminal_TASC

def crf(lifetime, i):
    return (i*np.power(1+i,lifetime)) / (np.power(1+i,lifetime)-1)

############################################################################################################################################


############################################################################################################################################
# TESTING

if __name__ == '__main__':
    pass