#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Cost function for biomass gasification plants in Germany
#################################################################################################################################################################################################################################################

import pandas as pd
import pyomo.environ as pyo
import numpy as np
from sklearn.linear_model import LinearRegression
from matplotlib import pyplot as plt
from supply_chain_optimization.edges_constraints.h2_compression import calculate_h2_compression

############################################################################################################################################
# CONSTRAINTS

def implement_biomass_gasification_cost_function(model, data, show_regression=False):
    """
    :param model: Pyomo model
    :param data: The dataset dict containing node and edge data
    :return: None
    """

    # Perform regression
    metadata = data['metadata']
    reg = calculate_regression_on_cost_function(metadata, test_mode=False, show_regression=show_regression)

    model.nodes_biomass_gasification_hub_cost_constraint_07 = pyo.Constraint(model.wc_shipping_terminal_indices_07, rule=lambda model, terminal: cost_function_pyomo(model, terminal, reg))


def cost_function_pyomo(model, terminal, reg):
    """
    Cost function for the biomass gasification plants in Germany (excluding timber supply costs)
    """

    expansion_decision = model.nodes_biomass_gasification_hub_expansion_decision_07[terminal]
    expansion_size_hydrogen_kta = model.nodes_biomass_gasification_hub_expansion_size_07[terminal]
    expansion_size_hydrogen_mw = expansion_size_hydrogen_kta / 26.298 * 100  # [kt/a H2 -> MW H2]

    # Annual costs [MNOK2024/a] = a * Plant size [MW] + b
    annualized_costs = reg.coef_[0] * expansion_size_hydrogen_mw + reg.intercept_

    return model.nodes_biomass_gasification_hub_cost_07[terminal] == annualized_costs * 1000 * expansion_decision


############################################################################################################################################
# HELPER FUNCTIONS

def biomass_gasification_cost_nonlinear(hydrogen_production_mw, metadata):
    """
    Nonlinear function to calculate annualized base costs (i.e., costs without timber supply costs)
    :param hydrogen_production_mw: Average hydrogen production in MW (LHV)
    :param metadata: Metadata Dataframe
    :return: Value for production costs excluding timber [MNOK2024/a]
    """

    hydrogen_production_kta = hydrogen_production_mw / 100 * 26.3 # Size of plant [kt/a hydrogen]
    hydrogen_production_kgh = hydrogen_production_mw / 100 * 3002 # Size of plant [kg/h hydrogen]
    size_factor = hydrogen_production_mw / 50

    annual_base_cost = 0 # [MNOK2024/a]
    main_plant_crf = crf(metadata['Biomass Gasification Main Plant Lifetime [a]'], metadata['WACC [%]']/100)
    h2_compressor_crf = crf(metadata['H2 Compressor Lifetime [a]'], metadata['WACC [%]']/100)

    ### CAPEX ###
    # Main Plant
    main_plant_TASC = metadata['Biomass Gasification (DE) Nominal TASC Main Plant [MEUR2024]'] * np.power(size_factor, 0.67) * 11.63 # [MNOK2024]
    annual_base_cost += main_plant_TASC * main_plant_crf # [MNOK2024/a]

    # H2 Compressor (10bar to 80bar)
    h2_compressor_TASC, _ = calculate_h2_compression(10, 80, hydrogen_production_kgh, metadata) # [MNOK2024]
    annual_base_cost += h2_compressor_TASC * h2_compressor_crf # [MNOK2024/a]

    ### OPEX ###

    # Linear OPEX (OPEX apart from wood chips, electricity, staff and maintenance)
    annual_base_cost += metadata['Biomass Gasification (DE) OPEX Nominal [tEUR2024/a]'] / 1000 * size_factor * 11.63 # [MNOK2024/a]

    # Electricity
    electricity_consumption = metadata['Biomass Gasification (DE) Electricity consumption [kWh/kgH2]'] * hydrogen_production_kta * 1000 * 1000 / 1000 # [MWh/a]
    annual_base_cost += electricity_consumption * metadata['Electricity price (DE) [€/MWh]'] * 11.63 / 1000 / 1000 # [MNOK2024/a]

    # Staff
    number_of_staff = np.ceil(0.775+11.22*size_factor)
    annual_base_cost += metadata['Biomass Gasification (DE) OPEX Staff [EUR2024/Staff]'] / 1000 / 1000 * number_of_staff * 11.63 # [MNOK2024/a]

    # Maintenance
    maintenance = (main_plant_TASC + h2_compressor_TASC) / 1.05 / 1.093 * 0.05 # [MNOK2024/a]
    annual_base_cost += maintenance

    return annual_base_cost # [MNOK2024/a]


def crf(lifetime, i):
    return (i*np.power(1+i,lifetime)) / (np.power(1+i,lifetime)-1)


def calculate_regression_on_cost_function(metadata, test_mode=False, show_regression=False):
    '''
    Calculate regression on the base costs (i.e., timber costs) of a biomass gasification plant in Germany
    :param metadata: The metadata dataset
    :param test_mode: Plot the regressions
    :return: Regression for the annualized base costs of biomass gasification plant in Germany
    '''

    biomass_gasification_size_list = np.linspace(metadata['Biomass Gasification SIZE - min [MW]'], metadata['Biomass Gasification SIZE - max [MW]'], 100)
    annualized_base_cost_list = []  # Annualized base costs (i.e., excluding timber costs) [MNOK2024/a]
    specific_base_cost_list = []  # Annualized specific base costs (i.e., excluding timber costs) for plotting in test mode [NOK2024/kg H2]

    for size_mw in biomass_gasification_size_list:
        cost = biomass_gasification_cost_nonlinear(size_mw, metadata)
        annualized_base_cost_list.append(cost)

        if test_mode:
            size_kga = size_mw / 100 * 26.3 * 1000 * 1000
            specific_base_cost_list.append(cost * 1000 * 1000 / size_kga)

    reg = LinearRegression().fit(biomass_gasification_size_list.reshape(-1, 1), np.array(annualized_base_cost_list))
    score = reg.score(biomass_gasification_size_list.reshape(-1, 1), np.array(annualized_base_cost_list))

    assert score > 0.98

    if test_mode or show_regression:
        # Plot regression for annual base costs
        fig, ax = plt.subplots(figsize=(9 * 0.7, 6 * 0.7))
        plt.plot(biomass_gasification_size_list, reg.predict(biomass_gasification_size_list.reshape(-1, 1)), color='#a02b92', label='Linear estimate')
        plt.scatter(biomass_gasification_size_list, annualized_base_cost_list, color='black', label='Nonlinear cost', s=1)
        plt.title('Base cost (i.e., cost excluding timber) for biomass gasification (DE)',  fontsize=8)
        plt.xlabel('Plant size [MW]', fontname='Inter')
        plt.ylabel('Annual base costs [MNOK2024/a]', fontname='Inter')
        linpred = 'y = {:.4f} * x + {:.4f} || R2 = {:.4f}'.format(reg.coef_[0], reg.intercept_, score)
        ax.annotate(linpred, xy=(0.95, 0.05), xycoords='axes fraction', fontsize=9, horizontalalignment='right', verticalalignment='bottom', fontname='Inter')
        plt.legend(loc='upper left')
        plt.tight_layout()
        plt.show()

        print('Score for regression for biomass gasification (DE): {:.2f}'.format(score))

    if test_mode:
        # Plot specific base costs
        fig, ax = plt.subplots(figsize=(9 * 0.7, 6 * 0.7))
        plt.scatter(biomass_gasification_size_list, specific_base_cost_list, color='#a02b92', label='Costs excluding timber', s=1)
        plt.xlabel('Plant size [MW]', fontname='Inter')
        plt.ylabel('Specific base costs [NOK2024/kg H2]', fontname='Inter')
        plt.legend(loc='upper right')
        plt.tight_layout()
        plt.show()

    return reg

############################################################################################################################################
# TESTING

if __name__ == '__main__':
    print('Testing cost function')

    metadata = pd.read_excel('../data/general/metadata.xlsx', index_col=0)['Value']

    expansion_size_hydrogen_mw = 400
    expansion_size_hydrogen_kta = expansion_size_hydrogen_mw / 100 * 26.3

    average_timber_price = 500 #  [NOK/m3]
    timber_supply_costs = average_timber_price / metadata['Timber density [kg/m3]'] / 1000  # [tNOK2024/kg timber]
    timber_supply_costs = timber_supply_costs * metadata['Biomass Gasification OPEX Timber consumption [kg/kgH2]']  # [tNOK2024/kgH2]
    timber_supply_costs = timber_supply_costs * 1000 * 1000 * expansion_size_hydrogen_kta  # [tNOK2024/a]

    reg = calculate_regression_on_cost_function(metadata, test_mode=True)
    annualized_costs_without_timber = (reg.coef_[0] * expansion_size_hydrogen_mw + reg.intercept_) * 1000 # [tNOK2024/a]
    annualized_costs_with_timber = annualized_costs_without_timber + timber_supply_costs # [tNOK2024/a]

    print('Specific production costs (excluding timber): {:.2f} [NOK2024/kg H2]'.format(annualized_costs_without_timber*1000/(expansion_size_hydrogen_kta*1000*1000)))
    print('Specific production costs (including timber): {:.2f} [NOK2024/kg H2] (p_timber = 500 NOK/m3]'.format(annualized_costs_with_timber * 1000 / (expansion_size_hydrogen_kta * 1000 * 1000)))