#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Cost function for biomass gasification plants in Norway
#################################################################################################################################################################################################################################################

import pandas as pd
import pyomo.environ as pyo
import numpy as np
from matplotlib import pyplot as plt
from sklearn.linear_model import LinearRegression
from supply_chain_optimization.edges_constraints.h2_compression import calculate_h2_compression
from supply_chain_optimization.functions.piecewise_linear_estimation import estimate as piecewise_linear_estimation

############################################################################################################################################
# CONSTRAINTS

def implement_biomass_gasification_cost_function(model, data, show_regression=False):
    """
    :param model: Pyomo model
    :param data: The dataset dict containing node and edge data
    :return: None
    """

    # Perform regression on nonlinear cost functions
    metadata = data['metadata']
    piecewise_reg_excluding_tt_terminal, reg_tt_terminal = calculate_regression_on_cost_function(metadata, test_mode=False, show_regression=show_regression)

    # Define piecewise linear cost constraint for base cost excluding tube trailer terminal
    for biomass_gasification_hub in model.gasification_hubs_indices_03:

        # Base cost
        piecewise_cost_i = pyo.Piecewise(
            model.nodes_biomass_gasification_hub_base_cost_03[biomass_gasification_hub],  # Dependent variable (cost) [tNOK2024/a]
            model.nodes_biomass_gasification_hub_expansion_size_03[biomass_gasification_hub],  # Independent variable (expansion size) [kt hydrogen per year]
            pw_pts=piecewise_reg_excluding_tt_terminal['breakpoints_kta'],  # Breakpoints for the piecewise function (constraint must be valid in the entire valid regin of the independent variable
            pw_constr_type='EQ',  # Enforce equality between cost and piecewise function
            f_rule=lambda model, expansion_size_kta: piecewise_base_cost_function(expansion_size_kta, piecewise_reg_excluding_tt_terminal)  # Cost rule
        )
        setattr(model, 'biomass_gasification_base_cost_constraint_03_' + biomass_gasification_hub, piecewise_cost_i)

    # Define total cost function (i.e., base cost excluding tube trailer terminal + tube trailer terminal + electricity)
    model.nodes_biomass_gasification_hub_cost_constraint_03 = pyo.Constraint(model.gasification_hubs_indices_03, rule=lambda model, gasification_hub: total_cost_function(model, gasification_hub, reg_tt_terminal, data))


def piecewise_base_cost_function(expansion_size_kta, piecewise_reg_excluding_tt_terminal):
    """
        Piecewise linear cost function for base costs for biomass gasification
        Annual base costs [MNOK2024/a] = a * Plant size [MW] + b
        :param expansion_size: Expansion size of the gasification hub [kt hydrogen per year]
        :return: Total cost [tNOK2024/a]
        """

    expansion_size_mw = expansion_size_kta / 26.2975 * 100
    a_values = piecewise_reg_excluding_tt_terminal['a_values']
    b_values = piecewise_reg_excluding_tt_terminal['b_values']
    breakpoints_mw = piecewise_reg_excluding_tt_terminal['breakpoints_mw']

    if expansion_size_mw <= breakpoints_mw[1]:
        cost = a_values[0] * expansion_size_mw + b_values[0]
    elif expansion_size_mw <= breakpoints_mw[2]:
        cost = a_values[1] * expansion_size_mw + b_values[1]
    else:
        cost = a_values[2] * expansion_size_mw + b_values[2]

    return cost * 1000


def total_cost_function(model, gasification_hub, reg_tt_terminal, data):
    """
    Cost function for the biomass gasification plants in Norway (excluding timber supply costs)
    """

    # Parameters
    metadata = data['metadata']
    node_data_gasification_hubs_03 = data['node_data_gasification_hubs_03']
    electricity_price = node_data_gasification_hubs_03.loc[gasification_hub, 'Electricity price [NOK2024/MWh]']

    # Variables
    expansion_size_main_plant = model.nodes_biomass_gasification_hub_expansion_size_03[gasification_hub] # [kt/a hydrogen]
    expansion_decision_main_plant = model.nodes_biomass_gasification_hub_expansion_decision_03[gasification_hub]

    expansion_size_tt_terminal = model.nodes_biomass_gasification_hub_ch2_tube_trailer_expansion_size_03[gasification_hub] # [kt/a hydrogen]
    expansion_size_tt_terminal_mw = expansion_size_tt_terminal / 26.298 * 100  # [kt/a H2 -> MW H2]
    expansion_decision_tt_terminal = model.nodes_biomass_gasification_hub_ch2_tube_trailer_expansion_decision_03[gasification_hub]

    # Base cost (i.e., annual costs without electricity and timber)
    base_cost_without_tt_terminal = model.nodes_biomass_gasification_hub_base_cost_03[gasification_hub]  # Base cost for the main plant without the tube trailer terminal excluding electricity costs [tNOK2024/a]
    base_cost_tt_terminal = (reg_tt_terminal.coef_[0] * expansion_size_tt_terminal_mw + reg_tt_terminal.intercept_) * 1000  # Base cost for the tube trailer terminal excluding electricity costs [tNOK2024/a]
    base_cost = base_cost_without_tt_terminal * expansion_decision_main_plant + base_cost_tt_terminal * expansion_decision_tt_terminal # Base cost [tNOK2024/a]

    # Electricity costs
    electricity_costs = metadata['Biomass Gasification OPEX Electricity consumption Main Plant [kWh/kgH2]'] * expansion_size_main_plant * 1000*1000 / 1000 * electricity_price / 1000 # [tNOK2024/a]
    electricity_costs += metadata['Biomass Gasification OPEX Electricity consumption cH2 TT Terminal [kWh/kgH2]'] * expansion_size_tt_terminal * 1000 * 1000 / 1000 * electricity_price / 1000 # [tNOK2024/a]

    return model.nodes_biomass_gasification_hub_cost_03[gasification_hub] == base_cost + electricity_costs

############################################################################################################################################


############################################################################################################################################
# HELPER FUNCTIONS

def biomass_gasification_cost_nonlinear(main_plant_size_mw, tt_terminal_size_mw, metadata):
    """
    Nonlinear function to calculate annualized base costs (i.e., cost without timber and electricity) of a biomass gasification plant
    :param main_plant_size_mw: Average hydrogen production in MW (LHV)
    :param tt_terminal_size_mw: Size of the tube trailer terminal
    :param metadata: Metadata Dataframe
    :return: Annualized base costs for the entire plant [MNOK2024/a], Annualized base costs for the tube trailer terminal [MNOK2024/a]
    """

    main_plant_size_kta = main_plant_size_mw / 100 * 26.3 # Size of entire plant [kt hydrogen per year]
    main_plant_size_kgh = main_plant_size_mw / 100 * 3002 # Size of entire plant [kg/h hydrogen]
    tt_terminal_size_kgh = tt_terminal_size_mw / 100 * 3002 # Size of sending terminal for outgoing tube trailers [kg/h hydrogen]
    size_factor = main_plant_size_mw / 50

    annual_base_cost_entire_plant = 0 # Annualized base costs (i.e., cost without timber and electricity) [MNOK2024/a]
    main_plant_crf = crf(metadata['Biomass Gasification Main Plant Lifetime [a]'], metadata['WACC [%]']/100)
    h2_compressor_crf = crf(metadata['H2 Compressor Lifetime [a]'], metadata['WACC [%]']/100)

    ### CAPEX ##

    # Main Plant
    main_plant_TASC = metadata['Biomass Gasification Nominal TASC Main Plant [MNOK2024]'] * np.power(size_factor, 0.67) # [MNOK2024]
    annual_base_cost_entire_plant += main_plant_TASC * main_plant_crf # [MNOK2024/a]

    # H2 Compressor (10bar to 350bar)
    h2_compressor_TASC, _ = calculate_h2_compression(10, 350, main_plant_size_kgh, metadata) # [MNOK2024]
    annual_base_cost_entire_plant += h2_compressor_TASC * h2_compressor_crf # [MNOK2024/a]

    ### - cH2 Tube Trailer Terminal - ###

    annual_base_cost_tt_terminal = 0 # [MNOK2024/a]

    # H2 Storage Vessels
    storage_size = calculate_storage_demand(tt_terminal_size_mw, metadata) # [kg]
    storage_TASC = metadata['cH2 Storage TASC 350bar [tNOK2024/kg H2]'] / 1000 * storage_size  # [MNOK2024]
    storage_cycle_life = metadata['cH2 Storage Cycle Life [-]'] / (main_plant_size_kta * 1000 * 1000 / storage_size)  # [a]
    storage_life = min(storage_cycle_life, 20)  # [a]
    storage_crf = crf(storage_life, metadata['WACC [%]'] / 100)  # [-]

    if storage_size > 0.1:
        annual_base_cost_entire_plant += storage_TASC * storage_crf # [MNOK2024/a]
        annual_base_cost_tt_terminal += storage_TASC * storage_crf # [MNOK2024/a]

    # H2 Loading Unit (Needs to be replaced after H2 Precooler Lifetime)
    num_incoming_trucks = tt_terminal_size_kgh / metadata['cH2 Tube Trailer Payload [kg]']  # [Trucks/h]
    filling_time = metadata['cH2 Tube Trailer Filling Time [min]'] / 60  # [h]
    truck_throughput_per_port = 1 / filling_time  # [Trucks/h]
    number_of_receiving_ports = np.ceil(num_incoming_trucks / truck_throughput_per_port)  # [-]
    h2_loading_unit_TASC = metadata['cH2 Tube Trailer Loading Terminal TASC [MNOK2024]'] * number_of_receiving_ports  # [MNOK2024]

    annual_base_cost_entire_plant += h2_loading_unit_TASC * crf(metadata['H2 Precooler Lifetime [a]'], metadata['WACC [%]'] / 100)  # [MNOK2024/a]
    annual_base_cost_tt_terminal += h2_loading_unit_TASC * crf(metadata['H2 Precooler Lifetime [a]'], metadata['WACC [%]'] / 100)  # [MNOK2024/a]

    ### OPEX ###

    # Auxiliary OPEX
    annual_base_cost_entire_plant += metadata['Biomass Gasification OPEX Nominal Auxiliary [tNOK2024/a]'] / 1000 * size_factor # [MNOK2024/a]

    # Staff
    number_of_staff = np.ceil(0.775+11.22*size_factor)
    annual_base_cost_entire_plant += metadata['Biomass Gasification OPEX Staff [NOK2024/Staff]'] / 1000 / 1000 * number_of_staff # [MNOK2024/a]

    # Maintenance
    maintenance = (main_plant_TASC + h2_compressor_TASC + storage_TASC + h2_loading_unit_TASC) / 1.05 / 1.093 * 0.05 # [MNOK2024/a]
    annual_base_cost_entire_plant += maintenance

    return annual_base_cost_entire_plant, annual_base_cost_tt_terminal # [MNOK2024/a]


def calculate_regression_on_cost_function(metadata, test_mode=False, show_regression=False):
    """
    Calculate regression on the base costs (i.e., without electricity and timber costs) of a biomass gasification plant without tube trailer terminal and the base costs for the tube trailer terminal on sending side
    :param metadata: The metadata dataset
    :param test_mode: Plot the regression for the annual base cost of a biomass gasification plant without tube trailer terminal
    :return: Piecewise regression for biomass gasification plant without tube trailer terminal; Regression for the tube trailer terminal on sending side
    """

    plant_size_min_mw = metadata['Biomass Gasification SIZE - min [MW]']
    plant_size_max_mw = metadata['Biomass Gasification SIZE - max [MW]']

    plant_size_list_mw = np.linspace(plant_size_min_mw, plant_size_max_mw, 50)  # [MW H2]

    annual_base_cost_excluding_ttt_list = [] # Annual base costs for the gasification plant excluding outgoing tube trailer terminal # [MNOK2024/a]
    annual_base_cost_ttt_list = [] # Annual base cost for the H2 tube trailer terminal # [MNOK2024/a]
    specific_base_cost_list = []  # Annualized specific base costs (i.e., excluding timber costs but including an assumed electricity price of 500 NOK/MWh) for plotting in test mode [NOK2024/kg H2]

    # Calculate base costs without tube trailer terminal and base costs for tub trailer terminal
    for plant_size_mw in plant_size_list_mw:
        annual_base_cost_entire_plant, annual_base_cost_tt_terminal = biomass_gasification_cost_nonlinear(plant_size_mw, plant_size_mw, metadata)

        annual_base_cost_excluding_ttt = annual_base_cost_entire_plant - annual_base_cost_tt_terminal
        annual_base_cost_excluding_ttt_list.append(annual_base_cost_excluding_ttt)
        annual_base_cost_ttt_list.append(annual_base_cost_tt_terminal)

        if test_mode:
            plant_size_kga = plant_size_mw / 100 * 26.3 * 1000 * 1000
            plant_size_kta = plant_size_mw / 100 * 26.3

            # Assume an electricity price of 500 NOK/MWh for plotting in test mode
            electricity_price = 500 # [NOK/MWh]
            electricity_costs = metadata['Biomass Gasification OPEX Electricity consumption Main Plant [kWh/kgH2]'] * plant_size_kta * 1000*1000 / 1000 * electricity_price / 1000 / 1000 # [MNOK2024/a]
            electricity_costs += metadata['Biomass Gasification OPEX Electricity consumption cH2 TT Terminal [kWh/kgH2]'] * plant_size_kta * 1000 * 1000 / 1000 * electricity_price / 1000 / 1000 # [MNOK2024/a]

            annual_base_cost_entire_plant_testing = annual_base_cost_entire_plant + electricity_costs
            specific_base_cost_list.append(annual_base_cost_entire_plant_testing * 1000 * 1000 / plant_size_kga)

    ### Regression for base costs of plant excluding tube trailer terminal ###

    reg_excluding_tt_terminal = LinearRegression().fit(plant_size_list_mw.reshape(-1, 1), np.array(annual_base_cost_excluding_ttt_list))
    score = reg_excluding_tt_terminal.score(plant_size_list_mw.reshape(-1, 1), np.array(annual_base_cost_excluding_ttt_list))

    assert score > 0.98

    if test_mode:

        # Plot linear regression for base costs excluding tube trailer terminal (rejected)
        fig, ax = plt.subplots(figsize=(9 * 0.7, 6 * 0.7))
        plt.plot(plant_size_list_mw, reg_excluding_tt_terminal.predict(plant_size_list_mw.reshape(-1, 1)), color='#a02b92', label='Linear estimate')
        plt.scatter(plant_size_list_mw, annual_base_cost_excluding_ttt_list, color='black', label='Nonlinear cost', s=1)
        plt.title('Base cost (i.e., excluding electricity, timber, tube trailer terminal) for biomass gasification (NO)', fontsize=8)
        plt.xlabel('Plant size [MW]', fontname='Inter')
        plt.ylabel('Annual base costs [MNOK2024/a]', fontname='Inter')
        linpred = 'y = {:.4f} * x + {:.4f} || R2 = {:.4f}'.format(reg_excluding_tt_terminal.coef_[0], reg_excluding_tt_terminal.intercept_, score)
        ax.annotate(linpred, xy=(0.95, 0.05), xycoords='axes fraction', fontsize=9, horizontalalignment='right', verticalalignment='bottom', fontname='Inter')
        plt.legend(loc='upper left')
        plt.tight_layout()
        plt.show()

        print('Score for linear regression for biomass gasification (NO) excluding outgoing tube trailer terminal: {:.2f}\n'.format(score))

    if test_mode or show_regression:
        # Plot piecewise linear regression for base costs excluding tube trailer terminal (used)
        print('Score for piecewise linear regression for biomass gasification (NO) excluding outgoing tube trailer terminal:')

    breakpoints, a_values, b_values, score = piecewise_linear_estimation(plant_size_list_mw, annual_base_cost_excluding_ttt_list, 3, 'Plant size [MW]', 'Annual base costs excluding TTT [MNOK2024/a]',
                                                                         title='Base cost (i.e., excluding electricity and timber) for biomass gasification (NO)',
                                                                         print_output=test_mode or show_regression)
    breakpoints[0] = 0  # PWLF must be defined for the entire set of the terminal expansion size. Lower bound is implemented through size constraint

    piecewise_reg_excluding_tt_terminal = {}
    piecewise_reg_excluding_tt_terminal['breakpoints_mw'] = breakpoints
    piecewise_reg_excluding_tt_terminal['breakpoints_kta'] = list(np.array(breakpoints) / 100 * 26.2975)
    piecewise_reg_excluding_tt_terminal['a_values'] = a_values
    piecewise_reg_excluding_tt_terminal['b_values'] = b_values

    if test_mode:
        # Plot specific base costs
        fig, ax = plt.subplots(figsize=(9 * 0.7, 6 * 0.7))
        plt.scatter(plant_size_list_mw, specific_base_cost_list, color='black', label='Cost excluding timber, including p_el=500 NOK/MWh', s=1)
        plt.xlabel('Plant size [MW]', fontname='Inter')
        plt.ylabel('Specific base costs [NOK2024/kg H2]', fontname='Inter')
        plt.legend(loc='upper right')
        plt.tight_layout()
        plt.show()

    ### Regression for base costs of tube trailer terminal ###

    reg_tt_terminal = LinearRegression(fit_intercept=True).fit(plant_size_list_mw.reshape(-1, 1), np.array(annual_base_cost_ttt_list))
    score = reg_tt_terminal.score(plant_size_list_mw.reshape(-1, 1), np.array(annual_base_cost_ttt_list))

    assert score > 0.95

    if test_mode or show_regression:

        # Plot regression for tube trailer terminal
        fig, ax = plt.subplots(figsize=(9 * 0.7, 6 * 0.7))
        plt.plot(plant_size_list_mw, reg_tt_terminal.predict(plant_size_list_mw.reshape(-1, 1)), color='#a02b92', label='Linear estimate')
        plt.scatter(plant_size_list_mw, annual_base_cost_ttt_list, color='black', label='Nonlinear cost', s=1)
        plt.title('Cost for tube trailer terminal at biomass gasification (NO)', fontsize=8)
        plt.xlabel('Plant size [MW]', fontname='Inter')
        plt.ylabel('Annual costs [MNOK2024/a]', fontname='Inter')
        linpred = 'y = {:.4f} * x + {:.4f} || R2 = {:.4f}'.format(reg_tt_terminal.coef_[0], reg_tt_terminal.intercept_, score)
        ax.annotate(linpred, xy=(0.95, 0.05), xycoords='axes fraction', fontsize=9, horizontalalignment='right', verticalalignment='bottom', fontname='Inter')
        plt.legend(loc='upper left')
        plt.tight_layout()
        plt.show()

        print('\nScore for regression for biomass gasification (NO) for outgoing tube trailer terminal: {:.2f}'.format(score))

    return piecewise_reg_excluding_tt_terminal, reg_tt_terminal


def calculate_storage_demand(tt_terminal_throughput_mw, metadata):
    """
    Storage demand = Payload tube trailer * #num tube trailers loading simultaneously - produced hydrogen during loading
    :param tt_terminal_throughput_mw: Hydrogen exported via tube trailer
    :param metadata: The metadata dataset
    :return: Storage demand [kg hydrogen]
    """

    tt_terminal_throughput_kgh = tt_terminal_throughput_mw / 50 * 1501  # [kg/h]
    payload_tube_trailer = metadata['cH2 Tube Trailer Payload [kg]']

    num_incoming_trucks = tt_terminal_throughput_kgh / payload_tube_trailer  # [Trucks/h]
    filling_time = metadata['cH2 Tube Trailer Filling Time [min]'] / 60  # [h]
    truck_throughput_per_port = 1 / filling_time  # [Trucks/h]
    number_of_receiving_ports = np.ceil(num_incoming_trucks / truck_throughput_per_port)  # [-]

    produced_h2_during_loading = tt_terminal_throughput_kgh/60*filling_time # [kg hydrogen]
    storage_demand = payload_tube_trailer * number_of_receiving_ports - produced_h2_during_loading # [kg]

    return storage_demand


def crf(lifetime, i):
    return (i*np.power(1+i,lifetime)) / (np.power(1+i,lifetime)-1)

############################################################################################################################################


############################################################################################################################################
# TESTING

if __name__ == '__main__':
    print('Testing cost function:')

    metadata = pd.read_excel('../data/general/metadata.xlsx', index_col=0)['Value']

    expansion_size_hydrogen_mw = 50
    expansion_size_hydrogen_kta = expansion_size_hydrogen_mw / 100 * 26.3
    average_timber_price = 500  # [NOK/m3]

    timber_supply_costs = average_timber_price / metadata['Timber density [kg/m3]'] / 1000  # [tNOK2024/kg timber]
    timber_supply_costs = timber_supply_costs * metadata['Biomass Gasification OPEX Timber consumption [kg/kgH2]']  # [tNOK2024/kgH2]
    timber_supply_costs = timber_supply_costs * 1000 * 1000 * expansion_size_hydrogen_kta  # [tNOK2024/a]

    piecewise_reg_excluding_tt_terminal, reg_tt_terminal = calculate_regression_on_cost_function(metadata, test_mode=True)

    # Plant without tube trailer terminal
    annualized_costs_without_timber = piecewise_base_cost_function(expansion_size_hydrogen_kta, piecewise_reg_excluding_tt_terminal)  # [tNOK2024/a]
    annualized_costs_with_timber = annualized_costs_without_timber + timber_supply_costs  # [tNOK2024/a]

    print('\nPlant without tube trailer terminal:')
    print('Specific production costs (excluding timber, excluding electricity): {:.2f} [NOK2024/kg H2]'.format(annualized_costs_without_timber * 1000 / (expansion_size_hydrogen_kta * 1000 * 1000)))
    print('Specific production costs (including timber, excluding electricity): {:.2f} [NOK2024/kg H2] (p_timber = 500 NOK/m3]'.format(annualized_costs_with_timber * 1000 / (expansion_size_hydrogen_kta * 1000 * 1000)))

    # Plant with tube trailer terminal
    annualized_costs_without_timber += (reg_tt_terminal.coef_[0] * expansion_size_hydrogen_mw + reg_tt_terminal.intercept_) * 1000  # [tNOK2024/a]
    annualized_costs_with_timber = annualized_costs_without_timber + timber_supply_costs  # [tNOK2024/a]

    print('\nPlant with tube trailer terminal:')
    print('Specific production costs (excluding timber, excluding electricity): {:.2f} [NOK2024/kg H2]'.format(annualized_costs_without_timber * 1000 / (expansion_size_hydrogen_kta * 1000 * 1000)))
    print('Specific production costs (including timber, excluding electricity): {:.2f} [NOK2024/kg H2] (p_timber = 500 NOK/m3]'.format(annualized_costs_with_timber * 1000 / (expansion_size_hydrogen_kta * 1000 * 1000)))