#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Size constraints for wood chip shipping terminals
#################################################################################################################################################################################################################################################

import pyomo.environ as pyo

def implement_wood_chip_shipping_terminal_export_size_constraints(model, data):
    """
    Function to implement all the necessary constraints for the given nodes
    :param model: The pyomo model
    :return: None
    """

    # Data extraction
    node_name = 'wc_shipping_terminal_export'
    expansion_size = model.nodes_wood_chip_shipping_terminal_expansion_size_05 # Expansion size in kt/a timber
    plant_indices = model.wc_shipping_terminal_indices_05 # The indices for which the constraints apply
    emissions = model.nodes_wood_chip_shipping_terminal_emission_05 # Total CO2 emissions at the terminal [tCO2eq/a]
    import_indices = model.timber_resources_indices_01 # The indices of the nodes which are connected upstream
    export_indices = model.wc_shipping_terminal_indices_07 # The indices of the nodes which are connected downstream

    # Constraint: Node size [kt timber per year] = sum of timber in [kt timber / year]
    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: constraint_inflow(model, expansion_size, node_index, import_indices))
    setattr(model, node_name + '_expansion_size_constraint_IN', const)

    # Constraint: Node size [kt timber per year] = sum of timber out [kt timber / year]
    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: constraint_outflow(model,expansion_size, node_index, export_indices))
    setattr(model, node_name + '_expansion_size_constraint_OUT', const)

    # Constraint: Emission = f(size)
    const = pyo.Constraint(rule=lambda model: emissions == get_emissions(model, data))
    setattr(model, node_name + '_emission_constraint', const)


def constraint_inflow(model, expansion_size, node_index, import_indices):

    transport_edge = model.edges_timber_truck_amount_04
    timber_in = sum(transport_edge[import_node, node_index] for import_node in import_indices) # Inflow of timber in kt/a
    return expansion_size[node_index] == timber_in


def constraint_outflow(model, expansion_size, node_index, export_indices):

    transport_edge = model.edges_wood_chip_shipping_amount_06
    timber_out = sum(transport_edge[node_index, export_index] for export_index in export_indices) # Outflow of wood chips in kt/a

    return expansion_size[node_index] == timber_out  # [kt/a timber]


def get_emissions(model, data):
    """
    Emission factors:
        - Electricity (for wood chip production)
    :return: Total emission for the export terminals [tCO2eq/a]
    """

    total_emissions = 0 # [tCO2eq/a]
    emission_factors = data['emission_factors']

    for terminal in model.wc_shipping_terminal_indices_05:

        timber_supply = model.nodes_wood_chip_shipping_terminal_expansion_size_05[terminal] * 1000 * 1000  # [kg/a]

        electricity_consumption_specific = 52  # kJ/kg timber
        electricity_consumption_absolute = electricity_consumption_specific * timber_supply / 3600 / 1000  # [MWh/a]

        total_emissions += electricity_consumption_absolute * emission_factors['Electricity (NO) [kgCO2eq/MWh]'] / 1000

    return total_emissions