#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Project: PhD Project - Optimization Model
# Task: Size constraints for timber production
#################################################################################################################################################################################################################################################

import pyomo.environ as pyo

def implement_timber_production_size_constraints(model, data):
    """
    Function to implement all the necessary constraints for the given nodes
    :param model: The pyomo model
    :param data: The dataset dict containing node and edge data
    :return: None
    """

    # Data extraction
    node_name = 'timber_production'
    node_data_biomass_01 = data['node_data_timber_resources_01']
    production_sites = model.timber_resources_indices_01 # The indices for which the constraints apply
    expansion_size = model.nodes_biomass_production_site_expansion_size_01 # Expansion size [kt/a timber]
    emissions = model.nodes_biomass_production_site_emission_01 # CO2 emissions at timber production [tCO2eq/a]

    # Constraint: Node size [kt timber per year] == Node outflow [kt timber per year]
    const = pyo.Constraint(production_sites, rule=lambda model, node_index: constraint_outflow(model, expansion_size, node_index))
    setattr(model, node_name + '_expansion_size_constraint', const)

    # Constraint: Node size [kt timber per year] <= Node potential [kt timber per year]
    const = pyo.Constraint(production_sites, rule=lambda model, node_index: expansion_size[node_index] <= node_data_biomass_01.loc[node_index, 'Potential [kt/a]'])
    setattr(model, node_name + '_expansion_decision_constraint_UB', const)

    # Constraint: Fuel consumption = f(productivity, amount)
    const = pyo.Constraint(rule=lambda model: emissions == get_emissions(model, data))
    setattr(model, node_name + '_emission_constraint', const)


def constraint_outflow(model, expansion_size, biomass_production_site):

    export_to_gasification_hubs = sum(model.edges_timber_truck_amount_02[biomass_production_site, export_index] for export_index in model.gasification_hubs_indices_03)
    export_to_shipping_terminals = sum(model.edges_timber_truck_amount_04[biomass_production_site, export_index] for export_index in model.wc_shipping_terminal_indices_05)

    return expansion_size[biomass_production_site] == export_to_gasification_hubs + export_to_shipping_terminals


def get_emissions(model, data):
    """
    Emission factors:
        - Diesel for harvester
        - Diesel for forwarder
    :return: Total emission for timber production in Norway [tCO2eq/a]
    """

    metadata = data['metadata']
    node_data = data['node_data_timber_resources_01']
    timber_density = metadata['Timber density [kg/m3]']
    diesel_density = metadata['Diesel density [kg/m3]'] / 1000 # [kg/liter]
    emission_factors = data['emission_factors']

    diesel_consumption_absolute = 0 # [kg diesel per year]

    for biomass_production_site in model.timber_resources_indices_01:
        expansion_size = model.nodes_biomass_production_site_expansion_size_01[biomass_production_site] # Timber production amount [kt timber per year]

        diesel_consumption_specific = node_data.loc[biomass_production_site, 'Diesel Consumption [l/m3]'] * diesel_density # [kg diesel per m3 timber]
        diesel_consumption_specific = diesel_consumption_specific / timber_density # [kg diesel per kg timber]
        diesel_consumption_absolute += diesel_consumption_specific * expansion_size * 1000 * 1000 # [kg diesel per year]

    return diesel_consumption_absolute * emission_factors['Diesel [kgCO2eq/kg]'] / 1000 # [tCO2eq per year]