#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Size constraints for cH2 shipping terminals
#################################################################################################################################################################################################################################################

import pyomo.environ as pyo

def implement_ch2_import_shipping_terminal_size_constraints(model, data):
    """
    Function to implement all the necessary constraints for the given nodes
    :param model: The pyomo model
    :return: None
    """

    # Data extraction
    node_name = 'ch2_shipping_terminal_import'
    metadata = data['metadata']
    expansion_size = model.nodes_ch2_shipping_terminal_expansion_size_07 # Expansion size in kt/a hydrogen
    expansion_decision = model.nodes_ch2_shipping_terminal_expansion_decision_07  # Expansion decision boolean
    emissions = model.nodes_ch2_shipping_terminal_emission_07  # Total CO2 emissions at cH2 terminal [tCO2eq/a]
    plant_indices = model.cH2_shipping_terminal_indices_07 # The indices for which the constraints apply
    import_indices = model.cH2_shipping_terminal_indices_05 # The indices of the nodes which are connected upstream

    lower_bound = metadata['cH2 Shipping Terminal (import) SIZE - min [MW]'] / 100 * 26.2975  # [kt hydrogen per year]
    upper_bound = 1314.875  # [kt hydrogen per year]

    # Constraint: Node size [kt hydrogen per year] = sum of hydrogen in [kt hydrogen / year]
    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: constraint_inflow(model, expansion_size, node_index, import_indices))
    setattr(model, node_name + '_expansion_size_constraint_IN', const)

    # Constraint: Expansion decision
    # If expansion_decision == 0: expansion_size = 0
    # If expansion_decision == 1: expansion_size >= lower_bound AND expansion_size <= upper_bound

    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: expansion_size[node_index] >= lower_bound * expansion_decision[node_index])
    setattr(model, node_name + '_expansion_decision_constraint_LB', const)

    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: expansion_size[node_index] <= upper_bound * expansion_decision[node_index])
    setattr(model, node_name + '_expansion_decision_constraint_UB', const)

    # Constraint: Utilization (=num trips * terminal time per trip) <= 8760
    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: constraint_utilization(model, node_index, metadata))
    setattr(model, node_name + '_utilization_constraint', const)

    # Constraint: Emission = f(size)
    const = pyo.Constraint(rule=lambda model: emissions == get_emissions(model, data))
    setattr(model, node_name + '_emission_constraint', const)


def constraint_inflow(model, expansion_size, node_index, import_indices):
    transport_edges = [model.edges_ch2_shipping_amount_06]
    hydrogen_in = sum(sum(transport_edge[import_index, node_index] for import_index in import_indices) for transport_edge in transport_edges) # Inflow of hydrogen in kt/a

    return expansion_size[node_index] == hydrogen_in


def constraint_utilization(model, node_index, metadata):
    number_of_incoming_trips = sum(model.edges_ch2_shipping_num_trips_06[ch2_shipping_terminal_export, node_index] for ch2_shipping_terminal_export in model.cH2_shipping_terminal_indices_05)
    utilization = number_of_incoming_trips * metadata['cH2 Shipping Terminal (import) Emptying time [min]'] / 60  # [hours per year]

    return utilization <= 8760


def get_emissions(model, data):
    """
    Emission factors:
        - Electricity
    :return: Total emission for cH2 import terminals [tCO2eq/a]
    """

    total_emissions = 0 # [tCO2eq/a]
    metadata = data['metadata']
    emission_factors = data['emission_factors']

    for terminal in model.cH2_shipping_terminal_indices_07:

        expansion_size = model.nodes_ch2_shipping_terminal_expansion_size_07[terminal] * 1000 * 1000 # Main terminal size [kg hydrogen per year]
        electricity_consumption = expansion_size * metadata['cH2 Shipping Terminal (import) OPEX Electricity consumption [kWh/kgH2]'] / 1000 # [MWh/a]

        total_emissions += electricity_consumption * emission_factors['Electricity (DE) [kgCO2eq/MWh]'] / 1000

    return total_emissions