#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Size constraints for cH2 shipping terminals
#################################################################################################################################################################################################################################################

import pyomo.environ as pyo

def implement_ch2_export_shipping_terminal_size_constraints(model, data):
    """
    Function to implement all the necessary constraints for the given nodes
    :param model: The pyomo model
    :param data: The dataset dict containing node and edge data
    :return: None
    """

    # Data extraction
    node_name = 'ch2_shipping_terminal_export'
    metadata = data['metadata']
    expansion_size = model.nodes_ch2_shipping_terminal_expansion_size_05 # Expansion size in kt/a hydrogen
    expansion_decision = model.nodes_ch2_shipping_terminal_expansion_decision_05 # Expansion decision boolean
    plant_indices = model.cH2_shipping_terminal_indices_05 # The indices for which the constraints apply
    import_indices = model.gasification_hubs_indices_03 # The indices of the nodes which are connected upstream
    export_indices = model.cH2_shipping_terminal_indices_07 # The indices of the nodes which are connected downstream
    emissions = model.nodes_ch2_shipping_terminal_emission_05 # Total CO2 emissions at cH2 terminal [tCO2eq/a]
    lower_bound = metadata['cH2 Shipping Terminal (export) SIZE - min [MW]'] / 100 * 26.2975 # Minimum throughput [kt/a]
    upper_bound = 1000 / 100 * 26.2975 # Maximum throughout [kt/a] -> Upper bound is given through the utilization constraint in practise. Value serves as bigM for the constraint

    # Constraint: Node size [kt hydrogen per year] = sum of hydrogen in [kt hydrogen / year]
    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: constraint_inflow(model, expansion_size, node_index, import_indices))
    setattr(model, node_name + '_expansion_size_constraint_IN', const)

    # Constraint: Node size [kt hydrogen per year] = sum of hydrogen out [kt hydrogen / year]
    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: constraint_outflow(model, expansion_size, node_index, export_indices))
    setattr(model, node_name + '_expansion_size_constraint_OUT', const)

    # Constraint: Size of tube trailer terminal at shipping terminal [kt hydrogen per year] = sum of hydrogen in via tube trailer [kt hydrogen / year]
    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: constraint_tube_trailer_terminal_size(model, node_index, import_indices))
    setattr(model, node_name + '_expansion_size_constraint_IN_TT', const)

    # Constraint: Size of tube trailer terminal at shipping terminal [kt hydrogen per year] = 0 if expansion decision is 0
    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: model.nodes_ch2_shipping_terminal_ch2_tube_trailer_expansion_size_05[node_index] <= upper_bound * model.nodes_ch2_shipping_terminal_ch2_tube_trailer_expansion_decision_05[node_index])
    setattr(model, node_name + '_expansion_decision_constraint_IN_TT', const)

    # Constraint: Expansion decision
    # If expansion_decision == 0: expansion_size = 0
    # If expansion_decision == 1: expansion_size >= lower_bound AND expansion_size <= upper_bound

    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: expansion_size[node_index] >= lower_bound * expansion_decision[node_index])
    setattr(model, node_name + '_expansion_decision_constraint_LB', const)

    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: expansion_size[node_index] <= upper_bound * expansion_decision[node_index])
    setattr(model, node_name + '_expansion_decision_constraint_UB', const)

    # Constraint: Utilization (=num trips * terminal time per trip) <= 8760
    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: constraint_utilization(model, node_index, metadata))
    setattr(model, node_name + '_utilization_constraint', const)

    # Constraint: Emission = f(size)
    const = pyo.Constraint(rule=lambda model: emissions == get_emissions(model, data))
    setattr(model, node_name + '_emission_constraint', const)


def constraint_inflow(model, expansion_size, node_index, import_indices):

    transport_edges = [model.edges_ch2_tube_trailer_amount_04, model.edges_ch2_pipeline_amount_04]
    hydrogen_in = sum(sum(transport_edge[import_index, node_index] for import_index in import_indices) for transport_edge in transport_edges) # Inflow of hydrogen in kt/a

    return expansion_size[node_index] == hydrogen_in


def constraint_outflow(model, expansion_size, node_index, export_indices):

    transport_edges = [model.edges_ch2_shipping_amount_06]
    hydrogen_out = sum(sum(transport_edge[node_index, export_index] for export_index in export_indices) for transport_edge in transport_edges) # Outflow of hydrogen in kt/a

    return expansion_size[node_index] == hydrogen_out


def constraint_tube_trailer_terminal_size(model, node_index, import_indices):

    tube_trailer_in = sum(model.edges_ch2_tube_trailer_amount_04[import_index, node_index] for import_index in import_indices)
    return model.nodes_ch2_shipping_terminal_ch2_tube_trailer_expansion_size_05[node_index] == tube_trailer_in


def constraint_utilization(model, node_index, metadata):

    number_of_outgoing_trips = sum(model.edges_ch2_shipping_num_trips_06[node_index, ch2_shipping_terminal_import] for ch2_shipping_terminal_import in model.cH2_shipping_terminal_indices_07)
    utilization = number_of_outgoing_trips * metadata['cH2 Shipping Terminal (export) Filling time [min]'] / 60  # [hours per year]

    return utilization <= 8760


def get_emissions(model, data):
    """
    Emission factors:
        - Electricity
    :return: Total emission for cH2 export terminals [tCO2eq/a]
    """

    total_emissions = 0 # [tCO2eq/a]
    metadata = data['metadata']
    emission_factors = data['emission_factors']

    for terminal in model.cH2_shipping_terminal_indices_05:

        expansion_size_main_plant = model.nodes_ch2_shipping_terminal_expansion_size_05[terminal] * 1000 * 1000 # Main terminal size [kg hydrogen per year]
        electricity_consumption_main_plant = expansion_size_main_plant * metadata['cH2 Shipping Terminal (export) OPEX Electricity consumption Main Plant [kWh/kgH2]'] / 1000 # [MWh/a]

        expansion_size_ch2ttt = model.nodes_ch2_shipping_terminal_ch2_tube_trailer_expansion_size_05[terminal] * 1000 * 1000  # Tube Trailer terminal size [kg hydrogen per year]
        electricity_consumption_ch2ttt = expansion_size_ch2ttt * metadata['cH2 Shipping Terminal (export) OPEX Electricity consumption cH2 TT Terminal [kWh/kgH2]'] / 1000  # [MWh/a]

        total_emissions += (electricity_consumption_main_plant + electricity_consumption_ch2ttt) * emission_factors['Electricity (NO) [kgCO2eq/MWh]'] / 1000

    return total_emissions