#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Size constraints for biomass gasification
#################################################################################################################################################################################################################################################

import pyomo.environ as pyo

def implement_biomass_gasification_size_constraints(model, data):
    """
    Function to implement all the necessary constraints for the given nodes
    :param model: The pyomo model
    :param data: The dataset dict containing node and edge data
    :return: None
    """

    # Data extraction
    metadata = data['metadata']
    node_name = 'biomass_gasification_DE'

    expansion_size_hydrogen = model.nodes_biomass_gasification_hub_expansion_size_07 # Expansion size in kt/a hydrogen
    expansion_decision = model.nodes_biomass_gasification_hub_expansion_decision_07  # Expansion decision boolean
    plant_indices = model.wc_shipping_terminal_indices_07 # The indices for which the constraints apply
    import_indices = model.wc_shipping_terminal_indices_05 # The indices of the nodes which are connected upstream
    emissions = model.nodes_biomass_gasification_hub_emission_07  # Total CO2 emissions at H2 production [tCO2eq/a]

    lower_bound = metadata['Biomass Gasification SIZE - min [MW]'] / 100 * 26.2975  # [kt hydrogen per year]
    upper_bound = metadata['Biomass Gasification SIZE - max [MW]'] / 100 * 26.2975  # [kt hydrogen per year]

    # Constraint: Node size [kt hydrogen per year] = sum of timber in [kt timber / year] * conversion rate
    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: constraint_inflow(model, expansion_size_hydrogen, node_index, import_indices, metadata))
    setattr(model, node_name + '_expansion_size_constraint_IN', const)

    # Constraint: Expansion decision
    # If expansion_decision == 0: expansion_size = 0
    # If expansion_decision == 1: expansion_size >= lower_bound AND expansion_size <= upper_bound

    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: expansion_size_hydrogen[node_index] >= lower_bound * expansion_decision[node_index])
    setattr(model, node_name + '_expansion_decision_constraint_LB', const)

    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: expansion_size_hydrogen[node_index] <= upper_bound * expansion_decision[node_index])
    setattr(model, node_name + '_expansion_decision_constraint_UB', const)

    # Constraint: Emission = f(size)
    const = pyo.Constraint(rule=lambda model: emissions == get_emissions(model, data))
    setattr(model, node_name + '_emission_constraint', const)


def constraint_inflow(model, expansion_size, node_index, import_indices, metadata):

    conversion_rate = 1 / metadata['Biomass Gasification OPEX Timber consumption [kg/kgH2]']  # kt hydrogen out / kt timber in
    transport_edges = [model.edges_wood_chip_shipping_amount_06]
    timber_in = sum(sum(transport_edge[import_index, node_index] for import_index in import_indices) for transport_edge in transport_edges) # Inflow of timber in kt/a

    return expansion_size[node_index] == timber_in * conversion_rate


def get_emissions(model, data):
    """
    Emission factors:
        - Electricity
        - Natural Gas for heating
    :return: Total emission for cH2 production at biomass gasification plants in Germany [tCO2eq/a]
    """

    total_emissions = 0 # [tCO2eq/a]
    metadata = data['metadata']
    emission_factors = data['emission_factors']

    for gasification_hub in model.wc_shipping_terminal_indices_07:

        expansion_size = model.nodes_biomass_gasification_hub_expansion_size_07[gasification_hub] * 1000 * 1000 # Gasification hub size [kg hydrogen per year]
        electricity_consumption = expansion_size * metadata['Biomass Gasification (DE) Electricity consumption [kWh/kgH2]'] / 1000 # [MWh/a]
        heat_consumption = expansion_size * metadata['Biomass Gasification (DE) Heat consumption [kWh/kgH2]'] / 1000 # [MWh/a]

        total_emissions += electricity_consumption * emission_factors['Electricity (DE) [kgCO2eq/MWh]'] / 1000
        total_emissions += heat_consumption * emission_factors['Natural Gas (DE) [kgCO2eq/MWh]'] / 1000

    return total_emissions