#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Size constraints for biomass gasification
#################################################################################################################################################################################################################################################

import pyomo.environ as pyo

def implement_biomass_gasification_size_constraints(model, data):
    """
    Function to implement all the necessary constraints for the given nodes
    :param model: The pyomo model
    :param data: The dataset dict containing node and edge data
    :return: None
    """

    # Data extraction
    metadata = data['metadata']
    node_name = 'biomass_gasification_NO'
    expansion_size = model.nodes_biomass_gasification_hub_expansion_size_03 # Expansion size in kt/a hydrogen
    expansion_decision = model.nodes_biomass_gasification_hub_expansion_decision_03 # Expansion decision boolean
    plant_indices = model.gasification_hubs_indices_03 # The indices for which the constraints apply
    import_indices = model.timber_resources_indices_01 # The indices of the nodes which are connected upstream
    export_indices = model.cH2_shipping_terminal_indices_05 # The indices of the nodes which are connected downstream
    emissions = model.nodes_biomass_gasification_hub_emission_03 # Total CO2 emissions at H2 production [tCO2eq/a]

    lower_bound = metadata['Biomass Gasification SIZE - min [MW]'] / 100 * 26.2975 # [kt hydrogen per year]
    upper_bound = metadata['Biomass Gasification SIZE - max [MW]'] / 100 * 26.2975 # [kt hydrogen per year]

    # Constraint: Node size [kt hydrogen per year] = sum of timber in * conversion rate [kt hydrogen / year]
    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: constraint_inflow(model, expansion_size, node_index, import_indices, metadata))
    setattr(model, node_name + '_expansion_size_constraint_IN', const)

    # Constraint: Node size [kt hydrogen per year] = sum of hydrogen out [kt hydrogen / year]
    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: constraint_outflow(model,expansion_size, node_index, export_indices))
    setattr(model, node_name + '_expansion_size_constraint_OUT', const)

    # Constraint: Size of tube trailer terminal at gasification plant [kt hydrogen per year] = sum of hydrogen out via tube trailer [kt hydrogen / year]
    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: constraint_tube_trailer_terminal_size(model, node_index, export_indices))
    setattr(model, node_name + '_expansion_size_constraint_OUT_TT', const)

    # Constraint: Size of tube trailer terminal at gasification plant [kt hydrogen per year] = 0 if expansion decision is 0
    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: model.nodes_biomass_gasification_hub_ch2_tube_trailer_expansion_size_03[node_index] <= upper_bound * model.nodes_biomass_gasification_hub_ch2_tube_trailer_expansion_decision_03[node_index])
    setattr(model, node_name + '_expansion_decision_constraint_OUT_TT', const)

    # Constraint: Expansion decision
    # If expansion_decision == 0: expansion_size = 0
    # If expansion_decision == 1: expansion_size >= lower_bound AND expansion_size <= upper_bound

    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: expansion_size[node_index] >= lower_bound * expansion_decision[node_index])
    setattr(model, node_name + '_expansion_decision_constraint_LB', const)

    const = pyo.Constraint(plant_indices, rule=lambda model, node_index: expansion_size[node_index] <= upper_bound * expansion_decision[node_index])
    setattr(model, node_name + '_expansion_decision_constraint_UB', const)

    # Constraint: Emission = f(size)
    const = pyo.Constraint(rule=lambda model: emissions == get_emissions(model, data))
    setattr(model, node_name + '_emission_constraint', const)


def constraint_inflow(model, expansion_size, node_index, import_indices, metadata):

    conversion_rate = 1/metadata['Biomass Gasification OPEX Timber consumption [kg/kgH2]']  # kt hydrogen out / kt timber in
    transport_edge = model.edges_timber_truck_amount_02
    timber_in = sum(transport_edge[biomass_production_site, node_index] for biomass_production_site in import_indices) # Inflow of timber in kt/a
    return expansion_size[node_index] == timber_in * conversion_rate


def constraint_outflow(model, expansion_size, node_index, export_indices):

    transport_edges = [model.edges_ch2_tube_trailer_amount_04, model.edges_ch2_pipeline_amount_04]
    hydrogen_out = sum(sum(transport_edge[node_index, ch2_shipping_terminal] for ch2_shipping_terminal in export_indices) for transport_edge in transport_edges) # Outflow of hydrogen in kt/a

    return expansion_size[node_index] == hydrogen_out  # [kt/a hydrogen]


def constraint_tube_trailer_terminal_size(model, node_index, export_indices):

    tube_trailer_out = sum(model.edges_ch2_tube_trailer_amount_04[node_index, ch2_shipping_terminal] for ch2_shipping_terminal in export_indices)
    return model.nodes_biomass_gasification_hub_ch2_tube_trailer_expansion_size_03[node_index] == tube_trailer_out


def get_emissions(model, data):
    """
    Emission factors:
        - Electricity
    :return: Total emission for cH2 production at biomass gasification plants in Norway [tCO2eq/a]
    """

    total_emissions = 0 # [tCO2eq/a]
    metadata = data['metadata']
    emission_factors = data['emission_factors']

    for gasification_hub in model.gasification_hubs_indices_03:

        expansion_size_main_plant = model.nodes_biomass_gasification_hub_expansion_size_03[gasification_hub] * 1000 * 1000 # Gasification hub size [kg hydrogen per year]
        expansion_size_ttt = model.nodes_biomass_gasification_hub_ch2_tube_trailer_expansion_size_03[gasification_hub] * 1000 * 1000  # Tube trailer terminal size [kg hydrogen per year]

        electricity_consumption = expansion_size_main_plant * metadata['Biomass Gasification OPEX Electricity consumption Main Plant [kWh/kgH2]'] / 1000 # [MWh/a]
        electricity_consumption += expansion_size_ttt * metadata['Biomass Gasification OPEX Electricity consumption cH2 TT Terminal [kWh/kgH2]'] / 1000  # [MWh/a]

        total_emissions += electricity_consumption * emission_factors['Electricity (NO) [kgCO2eq/MWh]'] / 1000

    return total_emissions