#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Sensitivity analysis
#################################################################################################################################################################################################################################################

import os.path
import numpy as np
import pandas as pd
import pyomo.environ as pyo
from matplotlib import pyplot as plt
from matplotlib.font_manager import FontProperties

from supply_chain_optimization.functions.initialize_metadata import initialize_metadata
from supply_chain_optimization.optimizer import run_optimization
from supply_chain_optimization.analysis import perform_analysis
from supply_chain_optimization.edges_constraints.cH2_shipping import get_cost_per_trip_on_shipping_route


def crf(lifetime, i):
    return (i*np.power(1+i,lifetime)) / (np.power(1+i,lifetime)-1)


def perform_pareto_sensitivity_bar(data, pdir, n):
    """
    - Sensitivity analysis of cost/emissions tradeoff as a bar chart
    - Change hydrogen demand
    - Two branches: Optimal branch and enforced cH2
    :param data: The dataset containing node and edge data
    :param pdir: The path to parent directory
    :param n: The number of samples per sensitivity case
    :return: None
    """

    print('Running pareto sensitivity analysis:')

    specific_costs_wc = []  # Specific cost in the wood chip shipping branch [NOK/kg H2]
    specific_emissions_wc = []  # Specific emissions in the wood chip shipping branch [kgCO2eq/kg H2]
    specific_costs_ch2 = []  # Specific cost in the cH2 shipping branch [NOK/kg H2]
    specific_emissions_ch2 = []  # Specific emissions in the cH2 shipping branch [kgCO2eq/kg H2]
    final_hydrogen_demand_mw_list = np.linspace(100, 745, n)

    if os.path.exists(pdir + 'supply_chain_optimization/plots/sensitivity_analysis/pareto_analysis.feather'):
        print('Previous data found, reading these...')

        df_data = pd.read_feather(pdir + 'supply_chain_optimization/plots/sensitivity_analysis/pareto_analysis.feather')

        final_hydrogen_demand_mw_list = df_data['Final hydrogen demand [MW]'].tolist()
        specific_costs_wc = df_data['Specific costs wc [NOK/kg H2]'].tolist()
        specific_emissions_wc = df_data['Specific emissions wc [kgCO2eq/kg H2]'].tolist()
        specific_costs_ch2 = df_data['Specific costs ch2 [NOK/kg H2]'].tolist()
        specific_emissions_ch2 = df_data['Specific emissions ch2 [kgCO2eq/kg H2]'].tolist()

    else:
        ### Optimal branch (WC) ###

        for final_hydrogen_demand_mw in final_hydrogen_demand_mw_list:
            final_hydrogen_demand_kta = final_hydrogen_demand_mw / 100 * 26.2975
            data['metadata'] = initialize_metadata(data['metadata'], final_hydrogen_demand_mw, 600)

            print('\tSample for {} MW (1)'.format(final_hydrogen_demand_mw))

            try:
                model = run_optimization(data, final_hydrogen_demand_kta, print_details=False, enforce_cH2_export=False, show_regression=False)
                objective_value = pyo.value(model.total_cost(model))  # [tNOK2024/a]
                objective_value_specific = objective_value * 1000 / (final_hydrogen_demand_kta * 1000 * 1000)  # [NOK/kg H2]

                emission_values_absolute = np.array([pyo.value(model.nodes_biomass_production_site_emission_01),
                                                     pyo.value(model.nodes_biomass_gasification_hub_emission_03) + pyo.value(model.nodes_biomass_gasification_hub_emission_07),
                                                     pyo.value(model.edges_timber_truck_emissions),
                                                     pyo.value(model.edges_ch2_tube_trailer_emissions_04),
                                                     pyo.value(model.nodes_ch2_shipping_terminal_emission_05) + pyo.value(model.nodes_ch2_shipping_terminal_emission_07) + pyo.value(model.nodes_wood_chip_shipping_terminal_emission_05),
                                                     pyo.value(model.edges_ch2_shipping_fleet_emissions_06) + pyo.value(model.edges_wood_chip_shipping_fleet_emissions_06)])  # [tCO2eq/a]
                emission_values_specific = emission_values_absolute * 1000 * 1000 / (final_hydrogen_demand_kta * 1000 * 1000)  # [gCO2eq/kg H2]
                emission_specific = sum(emission_values_specific)

                wc_size = sum(pyo.value(model.nodes_biomass_gasification_hub_expansion_size_07[idx]) for idx in model.wc_shipping_terminal_indices_07)  # [kt hydrogen per year]
                share_of_wc = wc_size / final_hydrogen_demand_kta

                if share_of_wc < 0.9999:
                    print('cH2 shipping was expanded and sensitivity analysis is not correct')
                    raise Exception()

            except Exception as e:
                objective_value_specific = np.nan

            finally:
                specific_costs_wc.append(objective_value_specific)
                specific_emissions_wc.append(emission_specific / 1000)

        ### Suboptimal branch (cH2) ###

        for final_hydrogen_demand_mw in final_hydrogen_demand_mw_list:
            final_hydrogen_demand_kta = final_hydrogen_demand_mw / 100 * 26.2975
            data['metadata'] = initialize_metadata(data['metadata'], final_hydrogen_demand_mw, 600)

            print('\tSample for {} MW (2)'.format(final_hydrogen_demand_mw))

            try:
                model = run_optimization(data, final_hydrogen_demand_kta, print_details=False, enforce_cH2_export=True, show_regression=False)
                objective_value = pyo.value(model.total_cost(model))  # [tNOK2024/a]
                objective_value_specific = objective_value * 1000 / (final_hydrogen_demand_kta * 1000 * 1000)  # [NOK/kg H2]

                emission_values_absolute = np.array([pyo.value(model.nodes_biomass_production_site_emission_01),
                                                     pyo.value(model.nodes_biomass_gasification_hub_emission_03) + pyo.value(model.nodes_biomass_gasification_hub_emission_07),
                                                     pyo.value(model.edges_timber_truck_emissions),
                                                     pyo.value(model.edges_ch2_tube_trailer_emissions_04),
                                                     pyo.value(model.nodes_ch2_shipping_terminal_emission_05) + pyo.value(model.nodes_ch2_shipping_terminal_emission_07) + pyo.value(model.nodes_wood_chip_shipping_terminal_emission_05),
                                                     pyo.value(model.edges_ch2_shipping_fleet_emissions_06) + pyo.value(model.edges_wood_chip_shipping_fleet_emissions_06)])  # [tCO2eq/a]
                emission_values_specific = emission_values_absolute * 1000 * 1000 / (final_hydrogen_demand_kta * 1000 * 1000)  # [gCO2eq/kg H2]
                emission_specific = sum(emission_values_specific)

            except Exception as e:
                print('Ignoring this value')
                objective_value_specific = np.nan

            finally:
                specific_costs_ch2.append(objective_value_specific)
                specific_emissions_ch2.append(emission_specific / 1000)

    ### Plot ###
    fig, ax = plt.subplots(figsize=(9 * 0.9, 6 * 0.9))

    # Model results
    group_1_el_1 = plt.scatter(np.mean(specific_emissions_wc)*np.ones(len(specific_costs_wc)), specific_costs_wc, color='#a02b92', marker='D', label='wood chip shipping', s=20)
    group_1_el_2 = plt.scatter(np.mean(specific_emissions_ch2)*np.ones(len(specific_costs_ch2)), specific_costs_ch2, color='#a02b92', marker='o', label='cH$_2$ shipping', s=20)
    group_1 = [group_1_el_1, group_1_el_2]

    # Literature values
    group_2_el_1 = plt.scatter(11, 36.8, color='slategray', marker='v', label='SMR - Schwab, Adams', s=20)
    group_2_el_2 = plt.scatter(11, 42.1, color='slategray', marker='^', label='SMR - H$_2$ Observatory', s=20)
    group_2_el_3 = plt.scatter(11, 28.7, color='slategray', marker='>', label='SMR - H$_2$ Observatory (adjusted NG price)', s=20)
    group_2_el_4 = plt.scatter(11, 25.3, color='slategray', marker='<', label='SMR - Cornette, Blondeau', s=20)
    group_2_el_5 = plt.scatter(11, 20.7, color='slategray', marker='D', label='SMR - IEAGHG 2017', s=20)
    group_2 = [group_2_el_1, group_2_el_2, group_2_el_3, group_2_el_4, group_2_el_5]

    # EL grid
    group_3_el_1 = plt.scatter(24.653, 102.6, color='darkgoldenrod', marker='v', label='EPRI (low estimate)', s=20)
    group_3_el_2 = plt.scatter(24.653, 123.0, color='darkgoldenrod', marker='^', label='EPRI (high estimate)', s=20)
    group_3_el_3 = plt.scatter(24.653, 85.0, color='darkgoldenrod', marker='>', label='Frieden, Leker', s=20)
    group_3_el_4 = plt.scatter(24.653, 109.6, color='darkgoldenrod', marker='<', label='H2 Observatory', s=20)
    group_3 = [group_3_el_1, group_3_el_2, group_3_el_3, group_3_el_4]

    # EL wind
    group_4_el_1 = plt.scatter(0.609, 102.6, color='seagreen', marker='v', label='EPRI (low estimate)', s=20)
    group_4_el_2 = plt.scatter(0.609, 123.0, color='seagreen', marker='^', label='EPRI (high estimate)', s=20)
    group_4_el_3 = plt.scatter(0.609, 85.0, color='seagreen', marker='>', label='Frieden, Leker', s=20)
    group_4_el_4 = plt.scatter(0.609, 109.6, color='seagreen', marker='<', label='H2 Observatory', s=20)
    group_4 = [group_4_el_1, group_4_el_2, group_4_el_3, group_4_el_4]

    ### Add Bars for Group Means ###
    bar_width = 1.5

    # Group 1 mean (wood chip + cH2)
    mean_em_wc = np.mean(specific_emissions_wc)
    mean_cost_wc = np.mean(specific_costs_wc)
    ax.bar(mean_em_wc, mean_cost_wc, width=bar_width, color='#a02b92', alpha=0.3)

    mean_em_ch2 = np.mean(specific_emissions_ch2)
    mean_cost_ch2 = np.mean(specific_costs_ch2)
    ax.bar(mean_em_ch2, mean_cost_ch2, width=bar_width, color='#a02b92', alpha=0.3)

    # Group 2 mean (SMR)
    mean_em_2 = 11
    mean_cost_2 = np.mean([36.8, 42.1, 28.7, 25.3, 20.7])
    ax.bar(mean_em_2, mean_cost_2, width=bar_width, color='slategray', alpha=0.3)

    # Group 3 mean (EL grid)
    mean_em_3 = 24.653
    mean_cost_3 = np.mean([102.6, 123.0, 85.0, 109.6])
    ax.bar(mean_em_3, mean_cost_3, width=bar_width, color='darkgoldenrod', alpha=0.3)

    # Group 4 mean (EL wind)
    mean_em_4 = 0.609
    mean_cost_4 = np.mean([102.6, 123.0, 85.0, 109.6])
    ax.bar(mean_em_4, mean_cost_4, width=bar_width, color='seagreen', alpha=0.3)

    plt.xlabel(r'Specific emissions [kgCO$_2\text{eq}$/kg H$_2$]', fontname='Inter')
    plt.ylabel('Specific costs [NOK/kg H$_2$]', fontname='Inter')
    plt.title('Pareto analysis', fontsize=14)

    bold_font = FontProperties(weight='bold')

    leg1 = ax.legend(handles=group_1, loc="upper right", bbox_to_anchor=(9.500, 140), bbox_transform=ax.transData, title=r'Supply NO $\rightarrow$ DE', title_fontproperties=bold_font)
    leg2 = ax.legend(handles=group_2, loc="upper right", bbox_to_anchor=(23.500, 140), bbox_transform=ax.transData, title='Domestic grey H$_2$ (DE)', title_fontproperties=bold_font)
    leg3 = ax.legend(handles=group_3, loc="upper right", bbox_to_anchor=(23.500, 50), bbox_transform=ax.transData, title='Grid electrolysis (DE, today)', title_fontproperties=bold_font)
    leg4 = ax.legend(handles=group_4, loc="upper right", bbox_to_anchor=(23.500, 90), bbox_transform=ax.transData, title='Grid electrolysis (DE, theoretical)', title_fontproperties=bold_font)

    leg1._legend_box.align = "left"
    leg2._legend_box.align = "left"
    leg3._legend_box.align = "left"
    leg4._legend_box.align = "left"

    ax.add_artist(leg1)
    ax.add_artist(leg2)
    ax.add_artist(leg3)
    ax.add_artist(leg4)

    plt.ylim(bottom=0, top=140)
    plt.tight_layout()
    plt.show()

    path = pdir + 'supply_chain_optimization/plots/sensitivity_analysis/pareto_analysis.png'
    fig.savefig(path, dpi=600)

    data_out = {
        'Final hydrogen demand [MW]': final_hydrogen_demand_mw_list,
        'Specific costs wc [NOK/kg H2]': specific_costs_wc,
        'Specific emissions wc [kgCO2eq/kg H2]': specific_emissions_wc,
        'Specific costs ch2 [NOK/kg H2]': specific_costs_ch2,
        'Specific emissions ch2 [kgCO2eq/kg H2]': specific_emissions_ch2
    }

    df_out = pd.DataFrame(data_out)
    df_out.to_excel(pdir + 'supply_chain_optimization/plots/sensitivity_analysis/pareto_analysis.xlsx')
    df_out.to_feather(pdir + 'supply_chain_optimization/plots/sensitivity_analysis/pareto_analysis.feather')


def perform_pareto_sensitivity(data, pdir, n):
    """
    - Sensitivity analysis of cost/emissions tradeoff
    - Change hydrogen demand
    - Two branches: Optimal branch and enforced cH2
    :param data: The dataset containing node and edge data
    :param pdir: The path to parent directory
    :param n: The number of samples per sensitivity case
    :return: None
    """

    print('Running pareto sensitivity analysis:')

    specific_costs_wc = []  # Specific cost in the wood chip shipping branch [NOK/kg H2]
    specific_emissions_wc = []  # Specific emissions in the wood chip shipping branch [gCO2eq/kg H2]
    specific_costs_ch2 = []  # Specific cost in the cH2 shipping branch [NOK/kg H2]
    specific_emissions_ch2 = []  # Specific emissions in the cH2 shipping branch [gCO2eq/kg H2]
    final_hydrogen_demand_mw_list = np.linspace(100, 745, n)

    if os.path.exists(pdir + 'supply_chain_optimization/plots/sensitivity_analysis/pareto_analysis.feather'):
        print('Previous data found, reading these...')

        df_data = pd.read_feather(pdir + 'supply_chain_optimization/plots/sensitivity_analysis/pareto_analysis.feather')

        final_hydrogen_demand_mw_list = df_data['Final hydrogen demand [MW]'].tolist()
        specific_costs_wc = df_data['Specific costs wc [NOK/kg H2]'].tolist()
        specific_emissions_wc = df_data['Specific emissions wc [gCO2eq/kg H2]'].tolist()
        specific_costs_ch2 = df_data['Specific costs ch2 [NOK/kg H2]'].tolist()
        specific_emissions_ch2 = df_data['Specific emissions ch2 [gCO2eq/kg H2]'].tolist()

    else:
        ### Optimal branch (WC) ###

        for final_hydrogen_demand_mw in final_hydrogen_demand_mw_list:
            final_hydrogen_demand_kta = final_hydrogen_demand_mw / 100 * 26.2975

            print('\tSample for {} MW (1)'.format(final_hydrogen_demand_mw))

            try:
                model = run_optimization(data, final_hydrogen_demand_kta, print_details=False, enforce_cH2_export=False, show_regression=False)
                objective_value = pyo.value(model.total_cost(model))  # [tNOK2024/a]
                objective_value_specific = objective_value * 1000 / (final_hydrogen_demand_kta * 1000 * 1000)  # [NOK/kg H2]

                emission_values_absolute = np.array([pyo.value(model.nodes_biomass_production_site_emission_01),
                                                     pyo.value(model.nodes_biomass_gasification_hub_emission_03) + pyo.value(model.nodes_biomass_gasification_hub_emission_07),
                                                     pyo.value(model.edges_timber_truck_emissions),
                                                     pyo.value(model.edges_ch2_tube_trailer_emissions_04),
                                                     pyo.value(model.nodes_ch2_shipping_terminal_emission_05) + pyo.value(model.nodes_ch2_shipping_terminal_emission_07) + pyo.value(model.nodes_wood_chip_shipping_terminal_emission_05),
                                                     pyo.value(model.edges_ch2_shipping_fleet_emissions_06) + pyo.value(model.edges_wood_chip_shipping_fleet_emissions_06)])  # [tCO2eq/a]
                emission_values_specific = emission_values_absolute * 1000 * 1000 / (final_hydrogen_demand_kta * 1000 * 1000)  # [gCO2eq/kg H2]
                emission_specific = sum(emission_values_specific)

                wc_size = sum(pyo.value(model.nodes_biomass_gasification_hub_expansion_size_07[idx]) for idx in model.wc_shipping_terminal_indices_07)  # [kt hydrogen per year]
                share_of_wc = wc_size / final_hydrogen_demand_kta

                if share_of_wc < 0.9999:
                    print('cH2 shipping was expanded and sensitivity analysis is not correct')
                    raise Exception()

            except Exception as e:
                objective_value_specific = np.nan

            finally:
                specific_costs_wc.append(objective_value_specific)
                specific_emissions_wc.append(emission_specific)

        ### Suboptimal branch (cH2) ###

        for final_hydrogen_demand_mw in final_hydrogen_demand_mw_list:
            final_hydrogen_demand_kta = final_hydrogen_demand_mw / 100 * 26.2975

            print('\tSample for {} MW (2)'.format(final_hydrogen_demand_mw))

            try:
                model = run_optimization(data, final_hydrogen_demand_kta, print_details=False, enforce_cH2_export=True, show_regression=False)
                objective_value = pyo.value(model.total_cost(model))  # [tNOK2024/a]
                objective_value_specific = objective_value * 1000 / (final_hydrogen_demand_kta * 1000 * 1000)  # [NOK/kg H2]

                emission_values_absolute = np.array([pyo.value(model.nodes_biomass_production_site_emission_01),
                                                     pyo.value(model.nodes_biomass_gasification_hub_emission_03) + pyo.value(model.nodes_biomass_gasification_hub_emission_07),
                                                     pyo.value(model.edges_timber_truck_emissions),
                                                     pyo.value(model.edges_ch2_tube_trailer_emissions_04),
                                                     pyo.value(model.nodes_ch2_shipping_terminal_emission_05) + pyo.value(model.nodes_ch2_shipping_terminal_emission_07) + pyo.value(model.nodes_wood_chip_shipping_terminal_emission_05),
                                                     pyo.value(model.edges_ch2_shipping_fleet_emissions_06) + pyo.value(model.edges_wood_chip_shipping_fleet_emissions_06)])  # [tCO2eq/a]
                emission_values_specific = emission_values_absolute * 1000 * 1000 / (final_hydrogen_demand_kta * 1000 * 1000)  # [gCO2eq/kg H2]
                emission_specific = sum(emission_values_specific)

            except Exception as e:
                print('Ignoring this value')
                objective_value_specific = np.nan

            finally:
                specific_costs_ch2.append(objective_value_specific)
                specific_emissions_ch2.append(emission_specific)

    ### Plot ###
    fig, ax = plt.subplots(figsize=(9 * 0.9, 6 * 0.9))

    # Model results
    group_1_el_1 = plt.scatter(specific_costs_wc, specific_emissions_wc, color='#a02b92', marker='D', label='Supply from NO (wood chip shipping)', s=20)
    group_1_el_2 = plt.scatter(specific_costs_ch2, specific_emissions_ch2, color='#a02b92', marker='o',label='Supply from NO (cH$_2$ shipping)', s=20)
    group_1 = [group_1_el_1, group_1_el_2]

    # Literature values
    group_2_el_1 = plt.scatter(36.8, 11000, color='slategray', marker='v', label='SMR - Schwab, Adams', s=20)
    group_2_el_2 = plt.scatter(42.1, 11000, color='slategray', marker='^', label='SMR - H$_2$ Observatory', s=20)
    group_2_el_3 = plt.scatter(28.7, 11000, color='slategray', marker='>', label='SMR - H$_2$ Observatory (adjusted NG price)', s=20)
    group_2_el_4 = plt.scatter(25.3, 11000, color='slategray', marker='<', label='SMR - Cornette, Blondeau', s=20)
    group_2_el_5 = plt.scatter(20.7, 11000, color='slategray', marker='D', label='SMR - IEAGHG 2017', s=20)
    group_2 = [group_2_el_1, group_2_el_2, group_2_el_3, group_2_el_4, group_2_el_5]

    group_3_el_1 = plt.scatter(102.6, 24653, color='darkgoldenrod', marker='v', label='EL grid - EPRI (low estimate)', s=20)
    group_3_el_2 = plt.scatter(123.0, 24653, color='darkgoldenrod', marker='^', label='EL grid - EPRI (high estimate)', s=20)
    group_3_el_3 = plt.scatter(85.0, 24653, color='darkgoldenrod', marker='>', label='EL grid - Frieden, Leker', s=20)
    group_3_el_4 = plt.scatter(109.6, 24653, color='darkgoldenrod', marker='<', label='EL grid - H2 Observatory', s=20)
    group_3 = [group_3_el_1, group_3_el_2, group_3_el_3, group_3_el_4]

    group_4_el_1 = plt.scatter(102.6, 609, color='seagreen', marker='v', label='EL wind - EPRI (low estimate)', s=20)
    group_4_el_2 = plt.scatter(123.0, 609, color='seagreen', marker='^', label='EL wind - EPRI (high estimate)', s=20)
    group_4_el_3 = plt.scatter(85.0, 609, color='seagreen', marker='>', label='EL wind - Frieden, Leker', s=20)
    group_4_el_4 = plt.scatter(109.6, 609, color='seagreen', marker='<', label='EL wind - H2 Observatory', s=20)
    group_4 = [group_4_el_1, group_4_el_2, group_4_el_3, group_4_el_4]

    plt.ylabel('Specific emissions [gCO$_2eq$/kg H$_2$]', fontname='Inter')
    plt.xlabel('Specific costs [NOK/kg H$_2$]', fontname='Inter')
    plt.title('Pareto analysis', fontsize=14)

    leg1 = ax.legend(handles=group_1, loc="upper right", bbox_to_anchor=(125, 11000), bbox_transform=ax.transData)
    leg2 = ax.legend(handles=group_2, loc="upper left", bbox_to_anchor=(20, 23000), bbox_transform=ax.transData)
    leg3 = ax.legend(handles=group_3, loc="upper right", bbox_to_anchor=(125, 23000), bbox_transform=ax.transData)
    leg4 = ax.legend(handles=group_4, loc="upper right", bbox_to_anchor=(125, 17000), bbox_transform=ax.transData)

    ax.add_artist(leg1)
    ax.add_artist(leg2)
    ax.add_artist(leg3)
    ax.add_artist(leg4)

    plt.tight_layout()
    plt.show()

    path = pdir + 'supply_chain_optimization/plots/sensitivity_analysis/pareto_analysis.png'
    fig.savefig(path, dpi=600)

    data_out = {
        'Final hydrogen demand [MW]': final_hydrogen_demand_mw_list,
        'Specific costs wc [NOK/kg H2]': specific_costs_wc,
        'Specific emissions wc [gCO2eq/kg H2]': specific_emissions_wc,
        'Specific costs ch2 [NOK/kg H2]': specific_costs_ch2,
        'Specific emissions ch2 [gCO2eq/kg H2]': specific_emissions_ch2
    }

    df_out = pd.DataFrame(data_out)
    df_out.to_excel(pdir + 'supply_chain_optimization/plots/sensitivity_analysis/pareto_analysis.xlsx')
    df_out.to_feather(pdir + 'supply_chain_optimization/plots/sensitivity_analysis/pareto_analysis.feather')


def perform_sensitivity_of_specific_costs_on_hydrogen_demand(data, pdir, n):
    """
    - Sensitivity analysis on design (different supply chain structure in all sensitivity cases)
    - Change hydrogen demand
    - Two branches: Optimal branch and enforced cH2
    :param data: The dataset containing node and edge data
    :param pdir: The path to parent directory
    :param n: The number of samples per sensitivity case
    :return: None
    """

    print('Running sensitivity analysis on hydrogen demand:')

    specific_costs_wc = []  # Specific cost in the wood chip shipping branch [NOK/kg H2]
    specific_costs_ch2 = []  # Specific cost in the cH2 shipping branch [NOK/kg H2]
    final_hydrogen_demand_mw_list = np.linspace(100, 750, n)

    ### Optimal branch (WC) ###

    for final_hydrogen_demand_mw in final_hydrogen_demand_mw_list:
        final_hydrogen_demand_kta = final_hydrogen_demand_mw / 100 * 26.2975
        data['metadata'] = initialize_metadata(data['metadata'], final_hydrogen_demand_mw, 600)

        print('\tSample for {} MW (1)'.format(final_hydrogen_demand_mw))

        try:
            model = run_optimization(data, final_hydrogen_demand_kta, print_details=False, enforce_cH2_export=False, show_regression=False)
            objective_value = pyo.value(model.total_cost(model))  # [tNOK2024/a]
            objective_value_specific = objective_value * 1000 / (final_hydrogen_demand_kta * 1000 * 1000)  # [NOK/kg H2]

            wc_size = sum(pyo.value(model.nodes_biomass_gasification_hub_expansion_size_07[idx]) for idx in model.wc_shipping_terminal_indices_07) # [kt hydrogen per year]
            share_of_wc = wc_size / final_hydrogen_demand_kta

            if share_of_wc < 0.9999:
                print('cH2 shipping was expanded and sensitivity analysis is not correct')
                raise Exception()

        except Exception as e:
            objective_value_specific = np.nan

        finally:
            specific_costs_wc.append(objective_value_specific)

    ### Suboptimal branch (cH2) ###

    for final_hydrogen_demand_mw in final_hydrogen_demand_mw_list:
        final_hydrogen_demand_kta = final_hydrogen_demand_mw / 100 * 26.2975
        data['metadata'] = initialize_metadata(data['metadata'], final_hydrogen_demand_mw, 600)

        print('\tSample for {} MW (2)'.format(final_hydrogen_demand_mw))

        try:
            model = run_optimization(data, final_hydrogen_demand_kta, print_details=False, enforce_cH2_export=True, show_regression=False)
            objective_value = pyo.value(model.total_cost(model))  # [tNOK2024/a]
            objective_value_specific = objective_value * 1000 / (final_hydrogen_demand_kta * 1000 * 1000)  # [NOK/kg H2]

            developed_cH2_shipping_terminals_export = data['node_data_cH2_shipping_terminals_05'].loc[[idx for idx in model.cH2_shipping_terminal_indices_05 if pyo.value(model.nodes_ch2_shipping_terminal_expansion_size_05[idx]) > 0.01], :]
            if len(developed_cH2_shipping_terminals_export) > 1:
                print(developed_cH2_shipping_terminals_export)

        except Exception as e:
            print('Ignoring this value')
            objective_value_specific = np.nan

        finally:
            specific_costs_ch2.append(objective_value_specific)

    ### Plot ###
    fig, ax = plt.subplots(figsize=(9 * 0.7, 6 * 0.7))
    plt.scatter(final_hydrogen_demand_mw_list, specific_costs_wc, color='black', marker='.', label='Wood chip shipping', s=3)
    plt.scatter(final_hydrogen_demand_mw_list, specific_costs_ch2, color='#a02b92', marker='*', label='cH2 shipping', s=3)
    plt.xlabel('Final hydrogen demand [MW LHV]', fontname='Inter')
    plt.ylabel('Specific costs [NOK2024/kg H2]', fontname='Inter')

    # Annotation for Wood chip shipping
    ax.annotate(
        "Wood chip shipping",
        xy=(list(final_hydrogen_demand_mw_list)[-5], specific_costs_wc[-5]),  # last point
        xytext=(500, 37),  # position of the text box
        textcoords='data',
        arrowprops=dict(arrowstyle="->", color='black'),
        bbox=dict(boxstyle="round,pad=0.3", fc="white", ec="black", lw=0.5)
    )

    # Annotation for cH2 shipping
    ax.annotate(
        "cH$_2$ shipping",
        xy=(list(final_hydrogen_demand_mw_list)[-5], specific_costs_ch2[-5]),  # last point
        xytext=(500, 52),  # position of the text box
        textcoords='data',
        arrowprops=dict(arrowstyle="->", color='#a02b92'),
        bbox=dict(boxstyle="round,pad=0.3", fc="white", ec="#a02b92", lw=0.5)
    )

    plt.title('Sensitivity on hydrogen demand', fontsize=8)
    # plt.legend(loc='upper right')
    plt.tight_layout()
    plt.show()

    path = pdir + 'supply_chain_optimization/plots/sensitivity_analysis/final_hydrogen_demand.png'
    fig.savefig(path, dpi=600)


def perform_sensitivity_of_shipping_terminal_utilization_on_h2_demand(data, pdir, n):
    """
        - Sensitivity analysis on design (different supply chain structure in all sensitivity cases)
        - Change hydrogen demand
        - Two branches: Optimal branch and enforced cH2
        :param data: The dataset containing node and edge data
        :param pdir: The path to parent directory
        :param n: The number of samples per sensitivity case
        :return: None
        """

    print('Running sensitivity analysis of cH2 terminal utilization on hydrogen demand:')

    df_utilization_export = pd.DataFrame()
    df_utilization_import = pd.DataFrame()
    final_hydrogen_demand_mw_list = np.linspace(100, 700, n)

    ### Iteration over H2 demand ###

    for final_hydrogen_demand_mw in final_hydrogen_demand_mw_list:
        final_hydrogen_demand_kta = final_hydrogen_demand_mw / 100 * 26.2975
        data['metadata'] = initialize_metadata(data['metadata'], final_hydrogen_demand_mw, 600)

        print('\tSample for {} MW'.format(final_hydrogen_demand_mw))

        try:
            model = run_optimization(data, final_hydrogen_demand_kta, print_details=False, enforce_cH2_export=True, show_regression=False)

            for ch2_shipping_terminal in model.cH2_shipping_terminal_indices_05:
                number_of_outgoing_trips = sum(pyo.value(model.edges_ch2_shipping_num_trips_06[ch2_shipping_terminal, ch2_shipping_terminal_import]) for ch2_shipping_terminal_import in model.cH2_shipping_terminal_indices_07)
                utilization = number_of_outgoing_trips * data['metadata']['cH2 Shipping Terminal (export) Filling time [min]'] / 60  # [hours per year]

                df_utilization_export.loc[final_hydrogen_demand_mw, ch2_shipping_terminal] = utilization

            for shipping_terminal_import in model.cH2_shipping_terminal_indices_07:
                number_of_incoming_trips = sum(pyo.value(model.edges_ch2_shipping_num_trips_06[ch2_shipping_terminal_export, shipping_terminal_import]) for ch2_shipping_terminal_export in model.cH2_shipping_terminal_indices_05)
                utilization = number_of_incoming_trips * data['metadata']['cH2 Shipping Terminal (import) Emptying time [min]'] / 60  # [hours per year]

                df_utilization_import.loc[final_hydrogen_demand_mw, shipping_terminal_import] = utilization

        except Exception as e:
            print('Ignoring this value')

            for idx in data['node_data_cH2_shipping_terminals_05'].index.values.tolist():
                df_utilization_export.loc[final_hydrogen_demand_mw, idx] = np.nan

            for idx in data['node_data_cH2_shipping_terminals_07'].index.values.tolist():
                df_utilization_import.loc[final_hydrogen_demand_mw, idx] = np.nan

    df_utilization_export = df_utilization_export.loc[:, ~((df_utilization_export.isna() | (df_utilization_export < 1)).all(axis=0))] # Only keep entries with non-zero values
    df_utilization_import = df_utilization_import.loc[:, ~((df_utilization_import.isna() | (df_utilization_import < 1)).all(axis=0))]  # Only keep entries with non-zero values

    ### Plot ###

    fig, ax = plt.subplots(2,1, figsize=(9 * 0.7, 6 * 0.7))
    plt.subplot(2, 1, 1)
    colors = ['slateblue', 'teal', '#a02b92', 'olive', 'darkred', 'peru', 'black']
    color_idx = 0

    # Plot all expanded terminals
    for terminal in df_utilization_export.columns.values.tolist():
        column_vals = df_utilization_export.loc[:,terminal].values

        plt.plot(final_hydrogen_demand_mw_list, column_vals, color=colors[color_idx], label=terminal + ' (export)')
        color_idx += 1


    for terminal in df_utilization_import.columns.values.tolist():
        column_vals = df_utilization_import.loc[:,terminal].values

        plt.plot(final_hydrogen_demand_mw_list, column_vals, color=colors[color_idx], label=terminal + ' (import)')
        color_idx += 1

    plt.ylabel('Utilization [h/a]', fontname='Inter')
    plt.title('Sensitivity cH2 shipping terminal utilization', fontsize=8)
    plt.legend(loc='upper left', fontsize=8)

    # Plot average
    plt.subplot(2, 1, 2)
    df_utilization_export[df_utilization_export < 1] = np.nan
    df_utilization_export["Average"] = df_utilization_export.mean(axis=1, skipna=True)

    plt.plot(final_hydrogen_demand_mw_list, df_utilization_export.loc[:,"Average"], color='black', label='Average (export)')
    plt.xlabel('Final hydrogen demand [MW LHV]', fontname='Inter')
    plt.ylabel('Utilization [h/a]', fontname='Inter')
    plt.legend(loc='upper left', fontsize=8)
    plt.tight_layout()
    plt.show()

    path = pdir + 'supply_chain_optimization/plots/sensitivity_analysis/terminal_utilization_ch2.png'
    fig.savefig(path, dpi=600)


    print('Running sensitivity analysis of wc terminal utilization on hydrogen demand:')

    df_utilization_export = pd.DataFrame()
    df_utilization_import = pd.DataFrame()
    final_hydrogen_demand_mw_list = np.linspace(100, 700, n)

    ### Iteration over H2 demand ###

    for final_hydrogen_demand_mw in final_hydrogen_demand_mw_list:
        final_hydrogen_demand_kta = final_hydrogen_demand_mw / 100 * 26.2975
        data['metadata'] = initialize_metadata(data['metadata'], final_hydrogen_demand_mw, 600)

        print('\tSample for {} MW'.format(final_hydrogen_demand_mw))

        try:
            model = run_optimization(data, final_hydrogen_demand_kta, print_details=False, enforce_cH2_export=False, show_regression=False)

            for wc_shipping_terminal in model.wc_shipping_terminal_indices_05:
                number_of_outgoing_trips = sum(pyo.value(model.edges_wood_chip_shipping_num_trips_06[wc_shipping_terminal, shipping_terminal_import]) for shipping_terminal_import in model.wc_shipping_terminal_indices_07)
                utilization = number_of_outgoing_trips * data['metadata']['Timber ship transfer time export terminal [h]'] # [hours per year]

                df_utilization_export.loc[final_hydrogen_demand_mw, wc_shipping_terminal] = utilization

            for shipping_terminal_import in model.wc_shipping_terminal_indices_07:
                number_of_incoming_trips = sum(pyo.value(model.edges_wood_chip_shipping_num_trips_06[wc_shipping_terminal_export, shipping_terminal_import]) for wc_shipping_terminal_export in model.wc_shipping_terminal_indices_05)
                utilization = number_of_incoming_trips * data['metadata']['Timber ship transfer time import terminal [h]'] # [hours per year]

                df_utilization_import.loc[final_hydrogen_demand_mw, shipping_terminal_import] = utilization

        except Exception as e:
            print('Ignoring this value')

            for idx in data['node_data_wood_chip_shipping_terminals_05'].index.values.tolist():
                df_utilization_export.loc[final_hydrogen_demand_mw, idx] = np.nan

            for idx in data['node_data_cH2_shipping_terminals_07'].index.values.tolist():
                df_utilization_import.loc[final_hydrogen_demand_mw, idx] = np.nan

    df_utilization_export = df_utilization_export.loc[:, ~((df_utilization_export.isna() | (df_utilization_export < 1)).all(axis=0))]  # Only keep entries with non-zero values
    df_utilization_import = df_utilization_import.loc[:, ~((df_utilization_import.isna() | (df_utilization_import < 1)).all(axis=0))]  # Only keep entries with non-zero values

    ### Plot ###

    fig, ax = plt.subplots(2, 1, figsize=(9 * 0.7, 6 * 0.7))
    plt.subplot(2, 1, 1)
    count = 0

    # Plot all expanded terminals
    for terminal in df_utilization_export.columns.values.tolist():
        column_vals = df_utilization_export.loc[:, terminal].values

        if count == 0:
            plt.plot(final_hydrogen_demand_mw_list, column_vals, color='slateblue', label='Norwegian terminals (export)', linewidth=0.5)
            count += 1
        else:
            plt.plot(final_hydrogen_demand_mw_list, column_vals, color='slateblue', linewidth=0.5)

    for terminal in df_utilization_import.columns.values.tolist():
        column_vals = df_utilization_import.loc[:, terminal].values

        plt.plot(final_hydrogen_demand_mw_list, column_vals, color='teal', label=terminal + ' (import)')

    plt.ylabel('Utilization [h/a]', fontname='Inter')
    plt.legend(loc='upper left', fontsize=8)
    plt.title('Sensitivity wood chip shipping terminal utilization', fontsize=8)

    # Plot average
    plt.subplot(2, 1, 2)
    df_utilization_export[df_utilization_export < 1] = np.nan
    df_utilization_export["Average"] = df_utilization_export.mean(axis=1, skipna=True)

    plt.plot(final_hydrogen_demand_mw_list, df_utilization_export.loc[:, "Average"], color='black', label='Average (export)')
    plt.xlabel('Final hydrogen demand [MW LHV]', fontname='Inter')
    plt.ylabel('Utilization [h/a]', fontname='Inter')
    plt.legend(loc='upper left', fontsize=8)
    plt.tight_layout()
    plt.show()

    path = pdir + 'supply_chain_optimization/plots/sensitivity_analysis/terminal_utilization_wc.png'
    fig.savefig(path, dpi=600)


def compare_sc_branches_cost_emissions(data, pdir):
    """
    Calculate supply chain for a 400MW hydrogen demand. See what happens if cH2 shipping is enforced and plot breakdown of cost and emissions
    :param data: The dataset containing node and edge data
    :param pdir: The path to parent directory
    :return: None
    """

    final_hydrogen_demand_mw = 400
    final_hydrogen_demand_kta = final_hydrogen_demand_mw / 100 * 26.2975

    data['metadata'] = initialize_metadata(data['metadata'], final_hydrogen_demand_mw, 600)
    cost_categories = ['Timber Production', 'Gasification', 'Timber Truck', 'Tube Trailer', 'Shipping Terminals (export)', 'Shipping Terminals (import)', 'Shipping']

    # Optimal supply chain (timber)
    model = run_optimization(data, final_hydrogen_demand_kta, print_details=False, enforce_cH2_export=False, test_mode=False, show_regression=False)
    _, cost_values_specific_timber, emission_values_specific_timber = perform_analysis(data, model, 'BC', final_hydrogen_demand_kta, write_results=False, plot_nodes=False, get_duals=False, generate_bar_charts=False, pprint_variables=False, print_results=False)

    # Suboptimal supply chain (cH2)
    model = run_optimization(data, final_hydrogen_demand_kta, print_details=False, enforce_cH2_export=True, test_mode=False, show_regression=False)
    _, cost_values_specific_ch2, emission_values_specific_ch2 = perform_analysis(data, model, 'BC', final_hydrogen_demand_kta, write_results=False, plot_nodes=False, get_duals=False, generate_bar_charts=False, pprint_variables=False, print_results=False)

    ### Plot results for cost ###

    # Bar width and positions
    x = np.arange(len(cost_categories))  # label locations
    width = 0.35  # width of each bar

    # Create plot
    fig, ax = plt.subplots()
    ax.bar(x - width / 2, cost_values_specific_timber, width, label='Wood chip shipping', color='slategrey')
    ax.bar(x + width / 2, cost_values_specific_ch2, width, label=r'cH$_2$ shipping', color='thistle')

    plt.title('Breakdown of levelized costs of hydrogen (400MW)', fontsize=8)
    plt.ylabel('Specific cost [NOK/kg H2]', fontname='Inter')
    plt.grid(axis='y', linestyle='--', alpha=0.7)

    ax.set_xticks(x)
    ax.set_xticklabels(cost_categories)
    plt.xticks(rotation=45, ha='right')

    plt.legend(loc='upper right')
    plt.tight_layout()
    plt.show()

    path = pdir + 'supply_chain_optimization/plots/sensitivity_analysis/supply_chain_structure_cost.png'
    fig.savefig(path, dpi=600)

    ### Plot results for emissions ###

    # Bar width and positions
    emission_categories = ['Timber Production', 'Gasification', 'Timber Truck', 'Tube Trailer', 'Shipping Terminals', 'Shipping']
    x = np.arange(len(emission_categories))  # label locations
    width = 0.35  # width of each bar

    # Create plot
    fig, ax = plt.subplots()
    ax.bar(x - width / 2, emission_values_specific_timber, width, label='Wood chip shipping', color='slategrey')
    ax.bar(x + width / 2, emission_values_specific_ch2, width, label='cH2 shipping', color='thistle')

    plt.title('Breakdown of operational supply chain emissions (400MW)', fontsize=8)
    plt.ylabel('Specific emissions [gCO2eq/kg H2]', fontname='Inter')
    plt.grid(axis='y', linestyle='--', alpha=0.7)

    ax.set_xticks(x)
    ax.set_xticklabels(emission_categories)
    plt.xticks(rotation=45, ha='right')

    plt.legend(loc='upper right')
    plt.tight_layout()
    plt.show()

    path = pdir + 'supply_chain_optimization/plots/sensitivity_analysis/supply_chain_structure_emissions.png'
    fig.savefig(path, dpi=600)

    print('Total emissions (Timber): {:.2f} [gCO2eq/kgH2]'.format(sum(emission_values_specific_timber)))
    print('Total emissions (cH2): {:.2f} [gCO2eq/kgH2]'.format(sum(emission_values_specific_ch2)))
    print('Total costs (Timber): {:.2f} [NOK2024/kgH2]'.format(sum(cost_values_specific_timber)))
    print('Total costs (cH2): {:.2f} [NOK2024/kgH2]'.format(sum(cost_values_specific_ch2)))


def compare_sc_branches_shipping_costs(data, pdir):
    """
    Calculate supply chain for a 400MW hydrogen demand. See what happens if cH2 shipping is enforced and plot breakdown of shipping cost
    :param data: The dataset containing node and edge data
    :param pdir: The path to parent directory
    :return: None
    """

    final_hydrogen_demand_mw = 400
    final_hydrogen_demand_kta = final_hydrogen_demand_mw / 100 * 26.2975

    data['metadata'] = initialize_metadata(data['metadata'], final_hydrogen_demand_mw, 600)
    metadata = data['metadata']

    ### Optimal supply chain (timber) ###

    model = run_optimization(data, final_hydrogen_demand_kta, print_details=False, enforce_cH2_export=False, test_mode=False, show_regression=False)

    # General
    developed_timber_shipping_routes = [(terminal_export, terminal_import) for terminal_export in model.wc_shipping_terminal_indices_05 for terminal_import in model.wc_shipping_terminal_indices_07 if
                                        pyo.value(model.edges_wood_chip_shipping_amount_06[terminal_export, terminal_import]) > 0.01]
    number_of_ships = pyo.value(model.edges_wood_chip_shipping_fleet_num_ships_06)

    # Investment (ship)
    investment_costs_ship = metadata['Timber Ship Investment cost [MNOK2024]'] * crf(metadata['Timber Ship Lifetime [a]'], metadata['Transport Industry WACC [%]'] / 100) * number_of_ships  # [MNOK2024/a]

    # Fixed OPEX (ship)
    fixed_opex_ship = metadata['Timber Ship Fixed OPEX [MNOK2024/a]'] * number_of_ships  # [MNOK2024/a]

    # Route dependent costs
    fuel_costs = 0

    port_fees_export = 0  # [MNOK2024/a]
    port_fees_import = 0  # [MNOK2024/a]

    for route in developed_timber_shipping_routes:
        export_terminal_index = route[0]
        import_terminal_index = route[1]

        fuel_costs += pyo.value(model.edges_wood_chip_shipping_cost_06[export_terminal_index, import_terminal_index]) / 1000  # [MNOK2024/a]

        num_trips = pyo.value(model.edges_wood_chip_shipping_num_trips_06[export_terminal_index, import_terminal_index])  # [#trips per year]
        days_at_export_terminal = np.ceil(metadata['Timber ship transfer time export terminal [h]'] / 24)  # Fees are paid as per døgn
        ship_size_bt = metadata['Timber Ship Gross tonnage [-]']
        ship_deadweight = metadata['Timber Ship Deadweight tonnage [tonnes]']

        # Export terminal fees

        if export_terminal_index in ['Sortland', 'Tromsø']:
            time_fee = data['node_data_wood_chip_shipping_terminals_05'].loc[export_terminal_index, 'Time fees'] * days_at_export_terminal / 1000  # Absolute fee [tNOK2024/Trip]
        elif export_terminal_index == 'Larvik':
            time_fee = data['node_data_wood_chip_shipping_terminals_05'].loc[export_terminal_index, 'Time fees'] * days_at_export_terminal / 1000 * 215  # Length based fee [tNOK2024/Trip]
        else:
            time_fee = data['node_data_wood_chip_shipping_terminals_05'].loc[export_terminal_index, 'Time fees'] * days_at_export_terminal / 1000 * ship_size_bt  # Size based fee [tNOK2024/Trip]

        waterway_fee = data['node_data_wood_chip_shipping_terminals_05'].loc[export_terminal_index, 'Waterway fee'] * ship_size_bt / 1000  # [tNOK2024/Trip]
        quay_fee = data['node_data_wood_chip_shipping_terminals_05'].loc[export_terminal_index, 'Quay fee wood chips'] * ship_deadweight / 1000  # [tNOK2024/Trip]

        port_fees_export += (waterway_fee + time_fee + quay_fee) / 1000 * num_trips  # [MNOK2024/a]

        # Import terminal fee (Wilhelmshaven)

        quay_fee = 0.598 * ship_deadweight * 11.63 / 1000  # [tNOK2024/Trip]
        time_fee = 0.2716 * ship_size_bt * 11.63 / 1000  # Can increase by 20% if additional days in harbour [tNOK2024/Trip]

        port_fees_import += (time_fee + quay_fee) / 1000 * num_trips  # [MNOK2024/a]

    cost_values_absolute_timber = np.array([investment_costs_ship, fixed_opex_ship, fuel_costs, port_fees_export, port_fees_import])  # [MNOK2024/a]
    cost_values_specific_timber = cost_values_absolute_timber / final_hydrogen_demand_kta  # [NOK/kg H2]

    print('Number of ships (WC shipping): {:.0f}'.format(pyo.value(model.edges_wood_chip_shipping_fleet_num_ships_06)))
    print('Number of trips (WC shipping): {:.0f}'.format(sum(pyo.value(model.edges_wood_chip_shipping_num_trips_06[export_idx, import_idx]) for export_idx in model.wc_shipping_terminal_indices_05 for import_idx in model.wc_shipping_terminal_indices_07)))

    ### Suboptimal supply chain (cH2) ###

    model = run_optimization(data, final_hydrogen_demand_kta, print_details=False, enforce_cH2_export=True, test_mode=False, show_regression=False)

    developed_ch2_shipping_routes = [(ch2_shipping_terminal_export, ch2_shipping_terminal_import) for ch2_shipping_terminal_export in model.cH2_shipping_terminal_indices_05 for ch2_shipping_terminal_import in model.cH2_shipping_terminal_indices_07 if
                                     pyo.value(model.edges_ch2_shipping_amount_06[ch2_shipping_terminal_export, ch2_shipping_terminal_import]) > 0.01]
    number_of_ships = pyo.value(model.edges_ch2_shipping_fleet_num_ships_06)

    # Investment cost (ship)
    ship_deadweight_tonnage = metadata['cH2 Ship Deadweight Tonnage [tonnes] (internal)']

    capex_cargo_ship = -380 + 2.6 * 149.3 + 1.8055 * ship_deadweight_tonnage / 1000 - 0.01009 * np.power(ship_deadweight_tonnage / 1000, 2) + 0.0000189 * np.power(ship_deadweight_tonnage / 1000, 3)  # Mulligan 2007 [M$2007]
    capex_cargo_ship = capex_cargo_ship / 186.2 * 287.6 * 10.75  # Adjusted to 2024 using Producer Price Index by Industry: Ship and Boat Building (PCU3366133661) [MNOK2024]
    capex_modules = metadata['cH2 Ship TASC module [tNOK2024/module]'] * metadata['cH2 Ship Num Modules [-]'] / 1000  # [MNOK2024]
    capex = capex_cargo_ship + capex_modules  # [MNOK2024]

    investment_costs_ship = capex * crf(metadata['cH2 Ship Lifetime [a]'], metadata['WACC [%]'] / 100)  # [MNOK2024/a and ship]
    investment_costs_ship = investment_costs_ship * number_of_ships  # [MNOK2024/a]

    # Fixed OPEX (ship)
    fixed_opex_ship = metadata['cH2 Ship Fixed OPEX [MNOK2024/a]']  # Salary costs + Insurance [MNOK2024/a and ship]
    fixed_opex_ship += capex * 0.03  # Maintenance [MNOK2024/a and ship]
    fixed_opex_ship = fixed_opex_ship * number_of_ships  # [MNOK2024/a]

    # Fuel costs
    fuel_costs = 0
    for route in developed_ch2_shipping_routes:
        export_terminal_index = route[0]
        import_terminal_index = route[1]

        number_of_trips_on_route = pyo.value(model.edges_ch2_shipping_num_trips_06[export_terminal_index, import_terminal_index])  # [trips/a]
        cost_per_trip = get_cost_per_trip_on_shipping_route(export_terminal_index, import_terminal_index, data)  # [tNOK2024/trip]
        fuel_costs += cost_per_trip * number_of_trips_on_route / 1000

    # Port fees
    port_fees_export = 0  # Separate terminal, annualized investment costs
    port_fees_import = 0  # Seperate terminal, annualized investment costs

    cost_values_absolute_ch2 = np.array([investment_costs_ship, fixed_opex_ship, fuel_costs, port_fees_export, port_fees_import])  # [MNOK2024/a]
    cost_values_specific_ch2 = cost_values_absolute_ch2 / final_hydrogen_demand_kta # [NOK/kg H2]

    print('Number of ships (cH2 shipping): {:.0f}'.format(pyo.value(model.edges_ch2_shipping_fleet_num_ships_06)))
    print('Number of trips (cH2 shipping): {:.0f}'.format(sum(pyo.value(model.edges_ch2_shipping_num_trips_06[export_idx, import_idx]) for export_idx in model.cH2_shipping_terminal_indices_05 for import_idx in model.cH2_shipping_terminal_indices_07)))

    ### Plot results ###

    # Bar width and positions
    cost_categories = ['Investment (ship)', 'Fixed OPEX (ship)', 'Fuel costs', 'Port fees (export)', 'Port fees (import)']
    x = np.arange(len(cost_categories))  # label locations
    width = 0.35  # width of each bar

    # Create plot
    fig, ax = plt.subplots()
    ax.bar(x - width / 2, cost_values_specific_timber, width, label='Wood chip shipping', color='slategrey')
    ax.bar(x + width / 2, cost_values_specific_ch2, width, label=r'cH$_2$ shipping', color='thistle')

    plt.title('Breakdown of shipping costs (400MW)', fontsize=8)
    plt.ylabel('Specific cost [NOK/kg H2]', fontname='Inter')
    plt.grid(axis='y', linestyle='--', alpha=0.7)

    ax.set_xticks(x)
    ax.set_xticklabels(cost_categories)
    plt.xticks(rotation=45, ha='right')

    plt.legend(loc='upper right')
    plt.tight_layout()
    plt.show()

    path = pdir + 'supply_chain_optimization/plots/sensitivity_analysis/shipping_cost_comparison.png'
    fig.savefig(path, dpi=600)