#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Transport constraint for wood chip shipping
#################################################################################################################################################################################################################################################

import numpy as np
import pandas as pd
import pyomo.environ as pyo
from matplotlib import pyplot as plt

############################################################################################################################################
# CONSTRAINTS

def implement_wood_chip_shipping_cost_constraint(model, data):
    """
    Constraint Shipping costs = f(edge thickness)
    :param model: Pyomo model
    :param data: The dataset dict containing node and edge data
    :return: None
    """

    export_terminals_indices = model.wc_shipping_terminal_indices_05
    import_terminals_indices = model.wc_shipping_terminal_indices_07

    # Number of ships >= required ships (required ships = shipping amount * factor)
    model.wood_chip_shipping_num_ships_constraint_06 = pyo.Constraint(rule=lambda model: number_of_ships_constraint(model, data))

    # Number of annual trips (int) >= annual shipping amount / shipping amount per trip
    model.wood_chip_shipping_num_trips_constraint_06 = pyo.Constraint(export_terminals_indices, import_terminals_indices, rule=lambda model, export_terminal_index, import_terminal_index: number_of_trips_constraint(model, export_terminal_index, import_terminal_index, data))

    # Variable shipping cost = number of trips * cost per trip
    model.wood_chip_shipping_variable_cost_constraint_06 = pyo.Constraint(export_terminals_indices, import_terminals_indices, rule=lambda model, export_terminal_index, import_terminal_index: variable_cost_constraint(model, export_terminal_index,import_terminal_index, data))

    # Fixed shipping costs = number of ships * ship's annual fixed costs
    model.wood_chip_shipping_fixed_cost_constraint_06 = pyo.Constraint(rule=lambda model: fixed_cost_constraint(model, data))

    # Fleet emissions = sum(number of trips on route * trip emissions)
    model.wood_chip_shipping_fleet_emissions_constraint_06 = pyo.Constraint(rule=lambda model: model.edges_wood_chip_shipping_fleet_emissions_06 == get_fleet_emissions(model, data))


def number_of_ships_constraint(model, data):
    # Number of ships >= sum of required ships (required ships = linear function of shipping amount) on route A-B for all routes

    required_number_of_ships = 0 # Minimum number of ships in the fleet [-]

    for export_terminal_index in model.wc_shipping_terminal_indices_05:
        for import_terminal_index in model.wc_shipping_terminal_indices_07:
            shipping_amount = model.edges_wood_chip_shipping_amount_06[export_terminal_index, import_terminal_index] # [kt timber per year]
            number_of_ships_on_route = get_required_number_of_ships_on_route(export_terminal_index, import_terminal_index, shipping_amount, data) # float, number of ships on route A-B [-]
            required_number_of_ships += number_of_ships_on_route

    return model.edges_wood_chip_shipping_fleet_num_ships_06 >= required_number_of_ships


def number_of_trips_constraint(model, export_terminal_index, import_terminal_index, data):
    # Number of annual trips (int) >= annual shipping amount / shipping amount per trip

    metadata = data['metadata']
    shipping_amount = model.edges_wood_chip_shipping_amount_06[export_terminal_index, import_terminal_index] # [kt timber per year]
    ship_deadweight = metadata['Timber Ship Deadweight tonnage [tonnes]']
    number_of_trips_float = shipping_amount * 1000 / ship_deadweight

    return model.edges_wood_chip_shipping_num_trips_06[export_terminal_index, import_terminal_index] >= number_of_trips_float


def fixed_cost_constraint(model, data):
    # Fixed shipping costs = number of ships * ship's annual fixed costs

    number_of_ships = model.edges_wood_chip_shipping_fleet_num_ships_06 # [-]
    fixed_annual_cost_per_ship = get_fixed_annual_cost_per_ship(data['metadata']) # [tNOK2024/a]

    return model.edges_wood_chip_shipping_fleet_cost_06 == number_of_ships * fixed_annual_cost_per_ship


def variable_cost_constraint(model, export_terminal_index, import_terminal_index, data):
    # Variable shipping cost = number of trips * cost per trip

    number_of_trips = model.edges_wood_chip_shipping_num_trips_06[export_terminal_index, import_terminal_index] # [trips/a]
    cost_per_trip = get_cost_per_trip_on_shipping_route(export_terminal_index, import_terminal_index, data) # [tNOK2024/Trip]

    return model.edges_wood_chip_shipping_cost_06[export_terminal_index, import_terminal_index] == number_of_trips * cost_per_trip


def get_fleet_emissions(model, data):
    # Fleet emissions = f(number of trips on route, distance of trip on route for route in routes)

    total_emissions = 0  # [tCO2eq/a]
    emission_factors = data['emission_factors']
    metadata = data['metadata']

    for export_terminal_index in model.wc_shipping_terminal_indices_05:
        for import_terminal_index in model.wc_shipping_terminal_indices_07:

            number_of_trips_on_route = model.edges_wood_chip_shipping_num_trips_06[export_terminal_index, import_terminal_index]  # [trips/a]
            shipping_distance = data['edge_data_shipping_distance_06'].loc[export_terminal_index, import_terminal_index] * 2  # [km/trip]

            total_sailing_time = shipping_distance / metadata['Timber Ship Sailing Speed [km/day]']  # [days/trip]
            total_terminal_time = (metadata['Timber ship transfer time export terminal [h]'] + metadata['Timber ship transfer time import terminal [h]']) / 24  # [days/trip]

            fuel_consumption_during_sailing = total_sailing_time * calculate_fuel_consumption(metadata)  # [tonnes/trip]
            fuel_consumption_during_terminal = total_terminal_time * calculate_fuel_consumption(metadata) * 0.05  # [tonnes/trip]
            fuel_consumption = fuel_consumption_during_sailing + fuel_consumption_during_terminal  # [tonnes/trip]
            fuel_consumption = fuel_consumption * number_of_trips_on_route  # [tonnes/a]

            total_emissions += fuel_consumption * emission_factors['VLSFO [kgCO2eq/kg]']  # [tCO2eq/a]

    return total_emissions
############################################################################################################################################


############################################################################################################################################
# HELPER FUNCTIONS

def get_required_number_of_ships_on_route(export_terminal_index, import_terminal_index, shipping_amount, data):
    """
    Calculate the number of ships (float) in the fleet on the given shipping route to ship the stated shipping amount
    :param shipping_amount: Timber shipping amount [kt timber per year]
    :return: number of ships (float)
    """

    metadata = data['metadata']
    ship_deadweight = metadata['Timber Ship Deadweight tonnage [tonnes]']  # [tonnes timber / ship]
    number_of_trips_float = shipping_amount * 1000 / ship_deadweight  # [trips/a for the entire fleet]

    shipping_distance = data['edge_data_shipping_distance_06'].loc[export_terminal_index, import_terminal_index]  # [km]
    total_transfer_time = metadata['Timber ship transfer time export terminal [h]'] + metadata['Timber ship transfer time import terminal [h]']  # [h/trip]
    trip_time = shipping_distance * 2 / metadata['Timber Ship Sailing Speed [km/day]'] + total_transfer_time / 24  # [days/trip]
    annual_trips_per_ship = metadata['Ship Utilization [h/a]'] / (trip_time * 24)  # [trips/a]
    number_of_ships_float = number_of_trips_float / annual_trips_per_ship  # [trips per year and fleet] / [trips per year and ship] = [#ships/fleet]

    return number_of_ships_float


def get_fixed_annual_cost_per_ship(metadata):
    # Annual ship's cost = fixed value for all ships

    # CAPEX + Fixed OPEX
    annual_costs = metadata['Timber Ship Investment cost [MNOK2024]'] * crf(metadata['Timber Ship Lifetime [a]'], metadata['Transport Industry WACC [%]']/100) # [MNOK2024/a]
    annual_costs += metadata['Timber Ship Fixed OPEX [MNOK2024/a]']

    return annual_costs * 1000 # [tNOK2024/a]


def get_cost_per_trip_on_shipping_route(export_terminal_index, import_terminal_index, data):
    """
    Calculate the cost per trip (i.e., fuel costs) on the given shipping route
    1 Trip = Loading at export terminal, sailing to import terminal, unloading at import terminal and sailing back to export terminal

    :return: Cost per trip [tNOK2024/Trip]
    """

    metadata = data['metadata']
    trip_cost = 0 # [tNOK2024/Trip]

    shipping_distance = data['edge_data_shipping_distance_06'].loc[export_terminal_index, import_terminal_index]  # [km]
    total_transfer_time = metadata['Timber ship transfer time export terminal [h]'] + metadata['Timber ship transfer time import terminal [h]'] # [h/trip]
    trip_time = shipping_distance * 2 / metadata['Timber Ship Sailing Speed [km/day]'] + total_transfer_time / 24  # [days/trip]

    # Fuel costs during sailing
    fuel_costs_during_sailing = calculate_fuel_consumption(metadata) * metadata['Ship Fuel price (VLSFO) [tNOK2024/tonne]'] # [tNOK2024/day]
    sailing_costs_per_trip = fuel_costs_during_sailing * (trip_time - total_transfer_time / 24) # [tNOK2024/Trip)
    trip_cost += sailing_costs_per_trip

    # Fuel costs at terminals (export + import)
    fuel_costs_during_terminal = fuel_costs_during_sailing * 0.05 / 24 * total_transfer_time  # [tNOK2024/Trip]
    trip_cost += fuel_costs_during_terminal

    return trip_cost # [tNOK2024/Trip]


def calculate_fuel_consumption(metadata):
    """
    Calculates the fuel consumption according to Cepowski et al. 2007
    :param metadata: The metadata dataset
    :return: Fuel consumption [tonnes/day]
    """

    dwt = metadata['Timber Ship Deadweight tonnage [tonnes]'] # [tonnes]
    ship_speed = metadata['Timber Ship Sailing Speed [knots]'] # [knots]

    fuel_consumption = 87.108 * ((dwt * 2.58 / 1000 / 1000 - 0.04178) * 0.90469 + (ship_speed * 0.294118-3.52941) * 0.130693 + 0.14126)

    return fuel_consumption


def crf(lifetime, i):
    return (i*np.power(1+i,lifetime)) / (np.power(1+i,lifetime)-1)
############################################################################################################################################


if __name__ == '__main__':
    # Test functions with example trip
    data_dir = '../data/'

    data = {}
    metadata = pd.read_excel(data_dir + 'general/metadata.xlsx', index_col=0)['Value']
    data['metadata'] = metadata
    data['edge_data_shipping_distance_06'] = pd.read_feather(data_dir + 'edge_data_06_shipping_distance.feather')  # Index = Shipping terminal names (export), Columns = Shipping terminal names (import), Values = Distance [km]
    data['node_data_wood_chip_shipping_terminals_05'] = pd.read_feather(data_dir + 'node_data_05_wc_shipping_terminals_export.feather')  # Index = Timber shipping terminals (export)

    shipping_amount = 2000  # [kt timber per year]

    fixed_annual_cost_per_ship = get_fixed_annual_cost_per_ship(metadata) # [tNOK2024/a]

    number_of_trips_float = shipping_amount * 1000 / metadata['Timber Ship Deadweight tonnage [tonnes]'] # Required trips to ship the defined amount [trips/a]
    number_of_trips = np.ceil(number_of_trips_float) # [trips/a]

    number_of_ships_float = get_required_number_of_ships_on_route('Florø', 'Wilhelmshaven', shipping_amount, data)
    number_of_ships = np.ceil(number_of_ships_float)  # [-]

    cost_per_trip = get_cost_per_trip_on_shipping_route('Florø', 'Wilhelmshaven', data)  # [tNOK2024/Trip]

    cost_absolute = fixed_annual_cost_per_ship * number_of_ships + number_of_trips * cost_per_trip # [tNOK2024/a]
    cost_relative = cost_absolute * 1000 / (shipping_amount*1000) # [NOK/t timber]
    cost_relative_per_h2_equivalent = cost_relative / 1000 * metadata['Biomass Gasification OPEX Timber consumption [kg/kgH2]'] # [NOK/kg H2]

    print('Number of ships: {:.2f} -> {}'.format(number_of_ships_float, number_of_ships))
    print('Cost on route: {:.2f} [MNOK2024/a]'.format(cost_absolute/1000))
    print('Cost on route: {:.2f} [NOK2024/t timber]'.format(cost_relative))
    print('Cost on route: {:.2f} [NOK2024/kg hydrogen]'.format(cost_relative_per_h2_equivalent))


    ### Sensitivity of specific ship's cost on shipping distance ###
    # - Fully utilized ship

    shipping_distance_list = np.linspace(250, 20000, 20)  # One-way distance [km]
    specific_cost_ship_list = []  # NOK/t wood chips

    for shipping_distance_i in shipping_distance_list:

        ### COSTS PER TRIP ###
        cost_per_trip = 0  # [tNOK2024/Trip]

        total_transfer_time = metadata['Timber ship transfer time export terminal [h]'] + metadata['Timber ship transfer time import terminal [h]']  # [h/trip]
        trip_time = shipping_distance_i * 2 / metadata['Timber Ship Sailing Speed [km/day]'] + total_transfer_time / 24  # [days/trip]

        # Fuel costs during sailing
        fuel_costs_during_sailing = calculate_fuel_consumption(metadata) * metadata['Ship Fuel price (VLSFO) [tNOK2024/tonne]']  # [tNOK2024/day]
        sailing_costs_per_trip = fuel_costs_during_sailing * (trip_time - total_transfer_time / 24)  # [tNOK2024/Trip)
        cost_per_trip += sailing_costs_per_trip

        # Fuel costs at terminals (export + import)
        fuel_costs_during_terminal = fuel_costs_during_sailing * 0.05 / 24 * total_transfer_time  # [tNOK2024/Trip]
        cost_per_trip += fuel_costs_during_terminal

        ### ANNUAL TRIPS ###
        annual_trips_per_ship = metadata['Ship Utilization [h/a]'] / (trip_time * 24)  # [trips/a]

        shipping_amount = metadata['Timber Ship Deadweight tonnage [tonnes]'] * annual_trips_per_ship

        total_annual_cost_ship = cost_per_trip * annual_trips_per_ship + fixed_annual_cost_per_ship # [tNOK2024/a]
        specific_cost_ship = total_annual_cost_ship * 1000 / (shipping_amount) # NOK/t wood chips
        specific_cost_ship_list.append(specific_cost_ship)

    plt.plot(shipping_distance_list, specific_cost_ship_list, 'k-*')
    plt.title('Sensitivity of specific shipping cost', fontsize=8)
    plt.xlabel('One-way shipping distance [km]')
    plt.ylabel('Specific cost (ship) [NOK2024/t wood chips]')
    plt.show()

    print('Factor cost increase with shipping distance: {:.2f} [NOK/t per 100 km]'.format((specific_cost_ship_list[-1] - specific_cost_ship_list[0]) / (2000 / 100 - 250 / 100)))
    print('Shipping costs for {} km: {:.2f} NOK/t wood chips'.format(shipping_distance_list[-1], specific_cost_ship_list[-1]))