#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Transport constraint for inland timber transport
#################################################################################################################################################################################################################################################

import pyomo.environ as pyo

def implement_timber_truck_transport_constraint(model, data):
    """
    Constraint: Cost for the timber truck fleet
    :param model: Pyomo model
    :param data: The dataset dict containing node and edge data
    :return: None
    """

    metadata = data['metadata']
    total_driven_distance, total_operation_time, _, total_emissions = get_fleet_size(model, data) # [km/a], [h/a], []
    required_number_of_trucks = total_operation_time / metadata['Timber Truck Utilization [h/a]']

    model.timber_truck_cost_constraint = pyo.Constraint(rule=lambda model: model.edges_timber_truck_cost == get_fleet_costs(model, data, total_driven_distance/1000))
    model.timber_truck_num_trucks_constraint = pyo.Constraint(rule=lambda model: model.edges_timber_truck_num_trucks >= required_number_of_trucks)
    model.timber_truck_emissions_constraint = pyo.Constraint(rule=lambda model: model.edges_timber_truck_emissions == total_emissions)


def get_fleet_size(model, data):
    """
    Get the annual driven distance and operational hours of the entire fleet
    """

    metadata = data['metadata']
    emission_factors = data['emission_factors']
    payload = metadata['Timber Truck Payload [tonnes]'] # [tonnes timber]
    terminal_time = metadata['Timber Truck Terminal time [h]'] # Loading, unloading and extra time [h]

    total_driven_distance = 0 # [km/a]
    total_operation_time = 0 # [h/a]
    total_num_trips = 0 # [trips/a]
    total_emissions = 0 # [tCO2eq/a]

    # Transport from forest to gasification plant (02)
    for source in model.timber_resources_indices_01:
        for destination in model.gasification_hubs_indices_03:
            transport_amount = model.edges_timber_truck_amount_02[source, destination] # Transport [kt timber per year]
            trip_distance = data['edge_data_timber_trucking_distance_02'].loc[source, destination] / 1000 * 2 # [km/trip]
            trip_driving_time = data['edge_data_timber_trucking_driving_time_02'].loc[source, destination] / 3600 * 2 # [h/trip]

            trip_time = trip_driving_time + terminal_time # [h/trip]
            required_trips_on_route = transport_amount * 1000 / payload # [#trips per year] -> Actually integer value. However, for simplicity it is assumed that backhauling is possible

            diesel_consumption_driving_A2B = metadata['Timber Truck Fuel consumption (loaded) [l/100km]'] / 100 * trip_distance / 2 # Fuel consumption for driving from source to destination [l/trip]
            diesel_consumption_driving_B2A = metadata['Timber Truck Fuel consumption (empty) [l/100km]'] / 100 * trip_distance / 2  # Fuel consumption for driving from destination to source [l/trip]
            diesel_consumption_terminal = metadata['Timber Truck Fuel consumption (during terminal) [l/h]'] * terminal_time # Fuel consumption at terminal [l/trip]
            diesel_consumption = (diesel_consumption_driving_A2B + diesel_consumption_driving_B2A + diesel_consumption_terminal) / 1000 * metadata['Diesel density [kg/m3]'] # [kg/trip]
            trip_emissions = diesel_consumption * emission_factors['Diesel [kgCO2eq/kg]'] / 1000 # [tCO2eq/trip]

            total_driven_distance += trip_distance * required_trips_on_route # [km/a]
            total_operation_time += trip_time * required_trips_on_route # [h/a]
            total_emissions += trip_emissions * required_trips_on_route # [tCO2eq/a]
            total_num_trips += required_trips_on_route # [trips/a]

    # Transport from forest to timber shipping terminal (04)
    for source in model.timber_resources_indices_01:
        for destination in model.wc_shipping_terminal_indices_05:
            transport_amount = model.edges_timber_truck_amount_04[source, destination] # Transport [kt timber per year]
            trip_distance = data['edge_data_timber_trucking_distance_04'].loc[source, destination] / 1000 * 2 # [km/trip]
            trip_driving_time = data['edge_data_timber_trucking_driving_time_04'].loc[source, destination] / 3600 * 2 # [h/trip]

            trip_time = trip_driving_time + terminal_time # [h/trip]
            required_trips_on_route = transport_amount * 1000 / payload # [#trips per year] -> Actually integer value. However, for simplicity it is assumed that backhauling is possible

            diesel_consumption_driving_A2B = metadata['Timber Truck Fuel consumption (loaded) [l/100km]'] / 100 * trip_distance / 2  # Fuel consumption for driving from source to destination [l/trip]
            diesel_consumption_driving_B2A = metadata['Timber Truck Fuel consumption (empty) [l/100km]'] / 100 * trip_distance / 2  # Fuel consumption for driving from source to destination [l/trip]
            diesel_consumption_terminal = metadata['Timber Truck Fuel consumption (during terminal) [l/h]'] * terminal_time  # Fuel consumption at terminal [l/trip]
            diesel_consumption = (diesel_consumption_driving_A2B + diesel_consumption_driving_B2A + diesel_consumption_terminal) / 1000 * metadata['Diesel density [kg/m3]']  # [kg/trip]
            trip_emissions = diesel_consumption * emission_factors['Diesel [kgCO2eq/kg]'] / 1000  # [tCO2eq/trip]

            total_driven_distance += trip_distance * required_trips_on_route # [km/a]
            total_operation_time += trip_time * required_trips_on_route # [h/a]
            total_emissions += trip_emissions * required_trips_on_route # [tCO2eq/a]
            total_num_trips += required_trips_on_route # [trips/a]

    return total_driven_distance, total_operation_time, total_num_trips, total_emissions


def get_fleet_costs(model, data, annual_distance_fleet):
    """
    Link the total fleet costs (fixed + variable) to the annual driven distance and number of trucks
    :param annual_distance_fleet: Annual distance driven by the fleet [tkm/a]
    """

    metadata = data['metadata']
    num_trucks_in_fleet = model.edges_timber_truck_num_trucks

    # Annual fixed fleet costs
    # Annual CAPEX per truck = 1.641 * annual_distance_per_truck [tkm] + 722.25 [tNOK2024/a and truck]
    # Annual CAPEX per fleet = 1.641 * annual_distance_fleet [tkm] + 722.25 * num_trucks_in_fleet [tNOK2024/a and truck]

    annual_capex_fleet = 1.641 * annual_distance_fleet + 722.25 * num_trucks_in_fleet # [tNOK2024/a]
    annual_fixed_opex_per_truck = metadata['Timber Truck Annual Fixed OPEX [tNOK2024/a]'] # [tNOK2024/a and truck]
    annual_fixed_fleet_costs = annual_capex_fleet + annual_fixed_opex_per_truck * num_trucks_in_fleet # [tNOK2024/a]

    # Annual variable fleet costs
    annual_variable_fleet_cost = metadata['Timber Truck Variable Cost [NOK2024/km]'] / 1000 * annual_distance_fleet * 1000 # [tNOK2024/a]

    return annual_fixed_fleet_costs + annual_variable_fleet_cost # [tNOK2024/a]