#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Calculate the cost and power for H2 compression given pressure and flow
#################################################################################################################################################################################################################################################

from CoolProp.CoolProp import PropsSI
import pandas as pd
import numpy as np

def calculate_h2_compression(pressure_in, pressure_out, mass_flow, metadata, temperature_inlet = 273.15+20):
    """
    Calculates the TASC of H2 compression following Khan, M.A. The Techno-Economics of Hydrogen Compression
    :param pressure_in: Pressure level at inlet [bar]
    :param pressure_out: Pressure level at outlet [bar]
    :param mass_flow: Mass flow of hydrogen [kg/h]
    :param metadata: The metadata dataset
    :param temperature_inlet: Temperature of H2 at inlet [K]
    :return: TASC of H2 compressor [MNOK2024], Compressor rated power [kW]
    """

    # Power calculation
    stage_pressure_ratio = metadata['H2 Compressor Stage pressure ratio [-]']
    n_stages = np.ceil(np.log10(pressure_out/pressure_in)/np.log10(stage_pressure_ratio))

    isentropic_efficieny = metadata['H2 Compressor Isentropic efficiency [-]']
    temperature_outlet = temperature_inlet * (1+(np.power(pressure_out/pressure_in, (1.4-1)/(n_stages*1.4))-1)/isentropic_efficieny)

    pressure_avrg = 2/3 * (np.power(pressure_out,3)-np.power(pressure_in,3))/(np.power(pressure_out,2)-np.power(pressure_in,2))
    temperature_avrg = (temperature_outlet + temperature_inlet) / 2

    Z = PropsSI('Z', 'T', temperature_avrg,'P',pressure_avrg*100*1000, 'Hydrogen')
    mole_flow = mass_flow / 0.002 / 60 / 60 # [mol/s]

    compressor_actual_power = n_stages*(1.4/(1.4-1))*Z/isentropic_efficieny*temperature_inlet*mole_flow*8.314*(np.power(pressure_out/pressure_in, (1.4-1)/(n_stages*1.4))-1) / 1000 # [kW]
    compressor_rated_power = compressor_actual_power / 0.95 # Accounting for 0.95 motor efficiency [kW]

    # Cost calculation
    equipment_cost = 3093.3 * np.power(compressor_rated_power,0.8335) # [C$2019]
    installed_costs = equipment_cost * 2 # [C$2019]
    installed_costs = installed_costs * 1.39 * 8.86 / 1000 / 1000 # [MNOK2024]
    TASC = installed_costs * 1.4 * 1.093 # [MNOK2024]

    return TASC, compressor_rated_power


if __name__ == '__main__':

    metadata = pd.read_excel('../data/general/metadata.xlsx', index_col=0)['Value']

    calculate_h2_compression(20,70,50000/24,metadata,temperature_inlet=305.15)