#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Transport constraint for cH2 inland transport
#################################################################################################################################################################################################################################################

import pyomo.environ as pyo
import numpy as np

def implement_ch2_tube_trailer_transport_constraint(model, data):
    """
    Implement size and cost constraints
    :param model: Pyomo model
    :param data: The dataset dictionary containing node and edge data
    :return: None
    """

    # Setup
    metadata = data['metadata']
    total_driven_distance, total_operation_time, total_num_trips, total_emissions = get_fleet_size(model, data) # [km/a], [h/a], [], [tCO2eq/a]
    required_number_of_trucks = total_operation_time / metadata['cH2 Tube Trailer Utilization [h/a]']

    # Constraint: Number of trucks in fleet > required number of trucks
    model.ch2_tube_trailer_num_trucks_constraint = pyo.Constraint(rule=lambda model: model.edges_ch2_tube_trailer_num_trucks_04 >= required_number_of_trucks)

    # Constraint: Fleet costs = f(annual_distance_fleet, num_trucks_in_fleet)
    model.ch2_tube_trailer_cost_constraint = pyo.Constraint(rule=lambda model: model.edges_ch2_tube_trailer_cost_04 == get_fleet_costs_linear(model, total_driven_distance/1000))

    # Constraint: Fleet emissions = f(annual_distance_fleet, annual_number_of_trips)
    model.ch2_tube_trailer_emission_constraint = pyo.Constraint(rule=lambda model: model.edges_ch2_tube_trailer_emissions_04 == total_emissions)


def get_fleet_size(model, data):
    """
    Get the annual driven distance and operational hours of the entire fleet
    """

    metadata = data['metadata']
    emission_factors = data['emission_factors']
    payload = metadata['cH2 Tube Trailer Payload [kg]'] / 1000 # [tonnes cH2]
    terminal_time = (metadata['cH2 Tube Trailer Emptying Time [min]'] + metadata['cH2 Tube Trailer Filling Time [min]']+15)/60 # Loading, unloading and extra time [h]

    total_driven_distance = 0 # [km/a]
    total_operation_time = 0 # [h/a]
    total_num_trips = 0 # [trips/a]
    total_emissions = 0  # [tCO2eq/a]

    # Transport from gasification plant to shipping terminal
    for source in model.gasification_hubs_indices_03:
        for destination in model.cH2_shipping_terminal_indices_05:
            transport_amount = model.edges_ch2_tube_trailer_amount_04[source, destination] # Transport [kt cH2 per year]
            trip_distance = data['edge_data_cH2_trucking_distance_04'].loc[source, destination] / 1000 * 2 # [km/trip]
            trip_driving_time = data['edge_data_cH2_trucking_driving_time_04'].loc[source, destination] / 3600 * 2 # [h/trip]

            trip_time = trip_driving_time + terminal_time # [h/trip]
            required_trips_on_route = transport_amount * 1000 / payload # [#trips per year] -> Actually integer value. However, for simplicity it is assumed that backhauling is possible

            diesel_consumption = metadata['cH2 Tube Trailer Fuel consumption [l/100km]'] / 100 * trip_distance # Fuel consumption for driving [l/trip]
            diesel_consumption = diesel_consumption / 1000 * metadata['Diesel density [kg/m3]'] # [kg/trip]
            trip_emissions = diesel_consumption * emission_factors['Diesel [kgCO2eq/kg]'] / 1000 # [tCO2eq/trip]

            total_driven_distance += trip_distance * required_trips_on_route # [km/a]
            total_operation_time += trip_time * required_trips_on_route # [h/a]
            total_emissions += trip_emissions * required_trips_on_route # [tCO2eq/a]
            total_num_trips += required_trips_on_route # [trips/a]

    return total_driven_distance, total_operation_time, total_num_trips, total_emissions


def get_fleet_costs_linear(model, annual_distance_fleet):
    """
    Link the total fleet costs (fixed + variable) to the annual driven distance and number of trucks
    :param annual_distance_fleet: Annual distance driven by the fleet [tkm/a]
    """

    num_trucks_in_fleet = model.edges_ch2_tube_trailer_num_trucks_04

    # Annual fleet costs
    # Annual cost per truck = 4.9808 * annual_distance_per_truck [tkm] + 9078.4 [tNOK2024/a and truck]
    # Annual cost per fleet = 4.9808 * annual_distance_fleet [tkm] + 9078.4 * num_trucks_in_fleet [tNOK2024/a and truck]

    annual_cost_fleet = 4.9808 * annual_distance_fleet + 9078.4 * num_trucks_in_fleet  # [tNOK2024/a]

    return annual_cost_fleet  # [tNOK2024/a]


def get_fleet_costs_nonlinear(model, annual_distance_fleet):
    """
    Link the total fleet costs (fixed + variable) to the annual driven distance and number of trucks
    :param annual_distance_fleet: Annual distance driven by the fleet [tkm/a]
    """

    num_trucks_in_fleet = model.edges_ch2_tube_trailer_num_trucks_04 # [-]
    annual_distance_per_truck = annual_distance_fleet / num_trucks_in_fleet # [tkm/a]

    annual_cost_per_truck = 0.0137*np.power(annual_distance_per_truck,2)-2.9176*annual_distance_per_truck+9998.4 # [tNOK2024/a]
    annual_cost_fleet = annual_cost_per_truck * num_trucks_in_fleet # [tNOK2024/a]

    return annual_cost_fleet  # [tNOK2024/a]