#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Analyse the results from the supply chain optimization
#################################################################################################################################################################################################################################################

import webbrowser
import folium
import os
import pandas as pd
import pyomo.environ as pyo
import numpy as np
from matplotlib import pyplot as plt
import contextily as cx
from pyproj import Transformer
from edges_constraints.timber_truck import get_fleet_size as get_fleet_size_timber_truck
from edges_constraints.cH2_tube_trailer import get_fleet_size as get_fleet_size_cH2_tube_trailer
from edges_constraints.cH2_tube_trailer import get_fleet_costs_nonlinear as get_fleet_costs_nonlinear_cH2_tube_trailer
from edges_constraints.cH2_shipping import get_required_number_of_ships_on_route
from edges_constraints.wood_chip_shipping import get_required_number_of_ships_on_route as get_required_number_of_ships_on_route_timber

pd.set_option('display.max_rows', 50)
pd.set_option('display.max_columns', 50)
pd.set_option('display.width', 5000)
pd.set_option('display.max_colwidth', 500)

def perform_analysis(data, model, RUN_ID, final_hydrogen_demand_kta, write_results=True, plot_nodes=False, generate_bar_charts=False, pprint_variables = False, print_results=True, case_name=None):
    """
    Perform analysis on a converged model
    :param data: The dataset containing node and edge data
    :param model: The pyomo model
    :param RUN_ID: The RUN_ID for the model
    :param final_hydrogen_demand_kta: The final hydrogen demand [kt/a]
    :param write_results: If the results should be saved to a .txt file
    :param plot_nodes: If a folium plot showing the expanded nodes should be generated
    :param generate_bar_charts: If plots showing emissions and cost distributions should be generated
    :param pprint_variables: If the variables for expansion size should be displayed using pyomo.pprint
    :param print_results: If the results should be printed in the console
    :return: Objective value [tNOK2024/a], Cost components [NOK/kg H2], Emission components [gCO2eq/kg H2]
    """

    if model is None:
        return None

    assert case_name is not None if plot_nodes else True, 'State case name if plot_nodes = True!'

    print('### ANALYZING RESULTS ###')
    result_string = '### RESULTS FOR RUN ID {} ###\n\n'.format(RUN_ID)

    # Extract data
    metadata = data['metadata']
    node_data_biomass_01 = data['node_data_timber_resources_01']  # Index = Biomass nodes, Columns = Potential [1000m3], Production costs [NOK/t]
    edge_data_timber_transport_distance_02 = data['edge_data_timber_trucking_distance_02'] # Indices = Biomass Nodes, Columns = Gasification Hubs, Values = Distance [m]
    edge_data_timber_transport_distance_04 = data['edge_data_timber_trucking_distance_04']  # Indices = Biomass Nodes, Columns = Timber shipping terminals, Values = Distance [m]
    node_data_gasification_hubs_03 = data['node_data_gasification_hubs_03']  # Index = Gasification Hubs, Columns = size constraints (min/max), cost data
    edge_data_cH2_tube_trailer_distance_04 = data['edge_data_cH2_trucking_distance_04']  # Indices = Gasification Hubs, Columns = Shipping terminals, Values = Distance [m]
    node_data_cH2_shipping_terminals_05 = data['node_data_cH2_shipping_terminals_05']  # Index = cH2 Shipping Terminals, Columns = size constraints (min/max), cost data
    node_data_wood_chip_shipping_terminals_05 = data['node_data_wood_chip_shipping_terminals_05']  # Index = Timber Shipping Terminals
    edge_data_shipping_distance_06 = data['edge_data_shipping_distance_06']  # Index = Shipping terminal names (export), Columns = Shipping terminal names (import), Values = Distance [km]
    node_data_cH2_shipping_terminals_07 = data['node_data_cH2_shipping_terminals_07'] # Index = Shipping terminal names (import)
    node_data_wood_chip_shipping_terminals_07 = data['node_data_wood_chip_shipping_terminals_07'] # Index = Shipping terminal names (import)

    # Setup
    # Select nodes that are developed
    developed_timber_production_sites = [idx for idx in model.timber_resources_indices_01 if pyo.value(model.nodes_biomass_production_site_expansion_size_01[idx]) > 0.01]
    developed_timber_transport_routes_02 = [(biomass_production_site, biomass_gasification_hub) for biomass_production_site in developed_timber_production_sites for biomass_gasification_hub in model.gasification_hubs_indices_03 if
                                pyo.value(model.edges_timber_truck_amount_02[biomass_production_site, biomass_gasification_hub]) > 0.01]
    developed_timber_transport_routes_04 = [(biomass_production_site, timber_shipping_terminal) for biomass_production_site in developed_timber_production_sites for timber_shipping_terminal in model.wc_shipping_terminal_indices_05 if
                                            pyo.value(model.edges_timber_truck_amount_04[biomass_production_site, timber_shipping_terminal]) > 0.01]
    developed_biomass_gasification_hubs_03 = node_data_gasification_hubs_03.loc[[idx for idx in model.gasification_hubs_indices_03 if pyo.value(model.nodes_biomass_gasification_hub_expansion_size_03[idx])>0.01],:]
    developed_cH2_shipping_terminals_export = node_data_cH2_shipping_terminals_05.loc[[idx for idx in model.cH2_shipping_terminal_indices_05 if pyo.value(model.nodes_ch2_shipping_terminal_expansion_size_05[idx]) > 0.01], :]
    developed_wood_chip_shipping_terminals_export = node_data_wood_chip_shipping_terminals_05.loc[[idx for idx in model.wc_shipping_terminal_indices_05 if pyo.value(model.nodes_wood_chip_shipping_terminal_expansion_size_05[idx]) > 0.01], :]
    developed_cH2_shipping_terminals_import = node_data_cH2_shipping_terminals_07.loc[[idx for idx in model.cH2_shipping_terminal_indices_07 if pyo.value(model.nodes_ch2_shipping_terminal_expansion_size_07[idx]) > 0.01], :]
    developed_wood_chip_shipping_terminals_import = node_data_wood_chip_shipping_terminals_07.loc[[idx for idx in model.wc_shipping_terminal_indices_07 if pyo.value(model.nodes_wood_chip_shipping_terminal_cost_07[idx]) > 0.01], :]
    developed_biomass_gasification_hubs_07 = [idx for idx in model.wc_shipping_terminal_indices_07 if pyo.value(model.nodes_biomass_gasification_hub_expansion_size_07[idx]) > 0.01]

    # Setup total cost parameters [tNOK2024 per year]
    total_timber_production_costs = 0
    total_gasification_costs = 0
    total_timber_truck_costs = pyo.value(model.edges_timber_truck_cost)
    total_tube_trailer_costs = pyo.value(model.edges_ch2_tube_trailer_cost_04)
    total_shipping_terminal_costs_export = 0
    total_shipping_terminal_costs_import = 0
    total_shipping_costs = 0

    # Create map
    if plot_nodes:
        map_level_1 = folium.Map(tiles="cartodb positron")
        map_level_2 = folium.Map(tiles="cartodb positron")

    #######################################################################################################

    #######################################################################################################
    # Biomass production sites

    result_string += 'Timber production:\n'
    
    # Transform CRS for plotting
    if plot_nodes:
        transformer_25833_to_4326 = Transformer.from_crs("EPSG:25833", "EPSG:4326")
        node_data_biomass_01['centroid_4326'] = node_data_biomass_01.apply(lambda row: tuple(reversed(transformer_25833_to_4326.transform(row.geometry.x, row.geometry.y))), axis=1)

    for timber_production_site in developed_timber_production_sites:
        production_level = pyo.value(model.nodes_biomass_production_site_expansion_size_01[timber_production_site]) # [kt timber per year]
        production_limit = node_data_biomass_01.loc[timber_production_site, 'Potential [kt/a]'] # [kt timber per year]
        destinations = [biomass_gasification_hub for biomass_gasification_hub in model.gasification_hubs_indices_03 if pyo.value(model.edges_timber_truck_amount_02[timber_production_site,biomass_gasification_hub]) > 0.01]
        destinations.extend([terminal for terminal in model.wc_shipping_terminal_indices_05 if pyo.value(model.edges_timber_truck_amount_04[timber_production_site,terminal]) > 0.01])

        total_timber_production_costs += pyo.value(model.nodes_biomass_production_site_cost_01[timber_production_site]) # [tNOK2024 per year]

        if plot_nodes:
            tooltip_html = '''<b>Timber Production site</b><br>
                        Index: {}<br>
                        Production level: {:.2f} [kt timber/a]<br>
                        Production limit: {:.2f} [kt timber/a]<br>
                        Supplies to: {}'''.format(timber_production_site, production_level, production_limit, destinations)

            folium.Marker(location=tuple(reversed(node_data_biomass_01.loc[timber_production_site, 'centroid_4326'])), icon=folium.Icon(color='green', icon_color='green', icon='tree', prefix='fa'), tooltip=tooltip_html).add_to(map_level_1)

        result_string += ('\tTimber production site {}: {:.2f} / {:.2f} [kt timber/a] || '
                          '{:.2f} / {:.2f} [NOK/t timber] (production costs / market price)\n').format(timber_production_site, production_level, production_limit,
                                                                                                       node_data_biomass_01.loc[timber_production_site, 'Production costs [NOK/t]'],
                                                                                                       node_data_biomass_01.loc[timber_production_site, 'Timber price [NOK/m3]']/metadata['Timber density [kg/m3]']*1000)
    #######################################################################################################

    #######################################################################################################
    # Timber Truck Transport

    result_string += 'Timber Truck Transport\n'

    total_driven_distance, _, total_num_trips, total_emissions = get_fleet_size_timber_truck(model, data)
    total_driven_distance = pyo.value(total_driven_distance)
    total_num_trips = pyo.value(total_num_trips)
    num_trucks_in_fleet = pyo.value(model.edges_timber_truck_num_trucks)
    annual_distance_per_truck = total_driven_distance / num_trucks_in_fleet / 1000 # [tkm per year and truck]
    annual_transported_timber = 0 # [kt timber per year]

    for route in developed_timber_transport_routes_02:
        edge_thickness = pyo.value(model.edges_timber_truck_amount_02[route[0], route[1]]) # [kt timber per year]
        distance = edge_data_timber_transport_distance_02.loc[route[0], route[1]] / 1000  # [km]
        num_trips = np.ceil(edge_thickness * 1000 / metadata['Timber Truck Payload [tonnes]']) # [-]
        annual_transported_timber += edge_thickness

        result_string += '\tTimber transport from {} to {}: {:.2f} [kt/a] for {:.2f} [km] on {:.0f} trips\n'.format(route[0], route[1], edge_thickness, distance, num_trips)

    for route in developed_timber_transport_routes_04:
        edge_thickness = pyo.value(model.edges_timber_truck_amount_04[route[0], route[1]])  # [kt timber per year]
        distance = edge_data_timber_transport_distance_04.loc[route[0], route[1]] / 1000  # [km]
        num_trips = np.ceil(edge_thickness * 1000 / metadata['Timber Truck Payload [tonnes]'])  # [-]
        annual_transported_timber += edge_thickness

        result_string += '\tTimber transport from {} to {}: {:.2f} [kt/a] for {:.2f} [km] on {:.0f} trips\n'.format(route[0], route[1], edge_thickness, distance, num_trips)

    assert annual_distance_per_truck > 50 and annual_distance_per_truck < 350, 'WARNING: Cost function for the annual capex of the timber truck out of range! Annual distance per truck is {:.2f} km'.format(annual_distance_per_truck)

    result_string += '\tFleet operation: {:.2f} [Mkm] using {:.0f} trucks on {:.0f} trips with an average one-way distance of {:.0f} [km]\n'.format(total_driven_distance/1000/1000, num_trucks_in_fleet, total_num_trips, total_driven_distance/total_num_trips/2)
    result_string += '\tFleet operation: {:.2f} [tNOK2024/a] annual costs per truck and {:.2f} [NOK2024/t timber] average transport costs\n'.format(pyo.value(model.edges_timber_truck_cost)/num_trucks_in_fleet, pyo.value(model.edges_timber_truck_cost)*1000/(annual_transported_timber*1000))

    #######################################################################################################

    #######################################################################################################
    # Biomass gasification hubs (NO)

    result_string += 'Biomass Gasification:\n'
    
    if plot_nodes and len(developed_biomass_gasification_hubs_03)>0:
        transformer_25833_to_4326 = Transformer.from_crs("EPSG:25833", "EPSG:4326")
        developed_biomass_gasification_hubs_03['centroid_4326'] = developed_biomass_gasification_hubs_03.apply(lambda row: tuple(reversed(transformer_25833_to_4326.transform(row.geometry.x, row.geometry.y))), axis=1)

    for biomass_gasification_hub in developed_biomass_gasification_hubs_03.index.to_list():

        # Node data
        expansion_size = pyo.value(model.nodes_biomass_gasification_hub_expansion_size_03[biomass_gasification_hub]) # [kt hydrogen per year]
        expansion_size_min = metadata['Biomass Gasification SIZE - min [MW]'] / 100 * 26.2975  # [kt hydrogen per year]
        expansion_size_max = metadata['Biomass Gasification SIZE - max [MW]'] / 100 * 26.2975 # [kt hydrogen per year]
        timber_supply_level = sum([pyo.value(model.edges_timber_truck_amount_02[timber_production_site, biomass_gasification_hub]) for timber_production_site in model.timber_resources_indices_01])
        ch2_tube_trailer_terminal_size = pyo.value(model.nodes_biomass_gasification_hub_ch2_tube_trailer_expansion_size_03[biomass_gasification_hub]) # [kt hydrogen per year]
        production_cost_absolute = pyo.value(model.nodes_biomass_gasification_hub_cost_03[biomass_gasification_hub]) # [tNOK2024 per year]
        production_cost_specific = production_cost_absolute / expansion_size / 1000 # [tNOK2024 per year] / [kt hydrogen per year] / 1000 = [NOK2024 / kg hydrogen]

        total_gasification_costs += production_cost_absolute

        if plot_nodes:
            tooltip_html = '''<b>Biomass Gasification plant</b><br>
                    Index: {}<br>
                    Production level: {:.2f} [kt hydrogen/a]<br>
                    Production min: {:.2f} [kt hydrogen/a]<br>
                    Production max: {:.2f} [kt hydrogen/a]<br>
                    Supply level: {:.2f} [kt timber/a]<br>
                    Production costs: {:.2f} [NOK/kg]'''.format(biomass_gasification_hub, expansion_size, expansion_size_min, expansion_size_max, timber_supply_level, production_cost_specific)

            folium.Marker(location=tuple(reversed(developed_biomass_gasification_hubs_03.loc[biomass_gasification_hub,'centroid_4326'])), icon=folium.Icon(color='black', icon_color='white', icon='industry', prefix='fa'), tooltip=tooltip_html).add_to(map_level_1)

        result_string += ('\tSE hub {}: {:.2f} (min) / {:.2f} / {:.2f} (max) [kt hydrogen/a] @ {:.2f} [NOK/kg hydrogen] || Timber supply: {:.2f} || cH2 Tube Trailer Terminal Size: {:.2f} kt/a\n'
                          .format(biomass_gasification_hub, expansion_size_min, expansion_size, expansion_size_max, production_cost_specific, timber_supply_level, ch2_tube_trailer_terminal_size))
    #######################################################################################################

    #######################################################################################################
    # cH2 pipeline transport routes

    developed_ch2_pipeline_routes = [(biomass_gasification_hub, ch2_shipping_terminal) for ch2_shipping_terminal in model.cH2_shipping_terminal_indices_05 for biomass_gasification_hub in model.gasification_hubs_indices_03 if
                                         pyo.value(model.edges_ch2_pipeline_amount_04[biomass_gasification_hub, ch2_shipping_terminal]) > 0.01]

    result_string += 'cH2 Pipeline Transport:\n'

    for route in developed_ch2_pipeline_routes:
        edge_thickness = pyo.value(model.edges_ch2_pipeline_amount_04[route[0], route[1]])  # [kt hydrogen per year]
        result_string += '\tcH2 transport from {} to {}: {:.2f} [kt/a] for {:.2f} [km]\n'.format(route[0], route[1], edge_thickness, edge_data_cH2_tube_trailer_distance_04.loc[route[0], route[1]] / 1000)

    #######################################################################################################

    #######################################################################################################
    # cH2 Tube trailer transport routes

    developed_ch2_tube_trailer_routes = [(biomass_gasification_hub, ch2_shipping_terminal) for ch2_shipping_terminal in model.cH2_shipping_terminal_indices_05 for biomass_gasification_hub in model.gasification_hubs_indices_03 if
                                pyo.value(model.edges_ch2_tube_trailer_amount_04[biomass_gasification_hub,ch2_shipping_terminal]) > 0.01]

    result_string += 'cH2 Tube Trailer Transport:\n'
    annual_transported_ch2 = 0  # [kt cH2 per year]

    for route in developed_ch2_tube_trailer_routes:
        edge_thickness = pyo.value(model.edges_ch2_tube_trailer_amount_04[route[0], route[1]]) # [kt hydrogen per year]
        distance = edge_data_cH2_tube_trailer_distance_04.loc[route[0], route[1]] / 1000 # [km]
        num_trips = np.ceil(edge_thickness * 1000 * 1000 / metadata['cH2 Tube Trailer Payload [kg]']) # [-]
        annual_transported_ch2 += edge_thickness

        result_string += '\tcH2 transport from {} to {}: {:.2f} [kt/a] for {:.2f} [km] using {:.0f} trips\n'.format(route[0], route[1], edge_thickness, distance, num_trips)

    if len(developed_ch2_tube_trailer_routes)>0:
        total_driven_distance, _, total_num_trips, total_emissions = get_fleet_size_cH2_tube_trailer(model, data)  # [km/a], [h/a], []
        total_driven_distance = pyo.value(total_driven_distance)
        total_num_trips = pyo.value(total_num_trips)

        num_trucks_in_fleet = pyo.value(model.edges_ch2_tube_trailer_num_trucks_04)
        annual_distance_per_truck = total_driven_distance / num_trucks_in_fleet / 1000  # [tkm per year and truck]

        assert annual_distance_per_truck > 110 and annual_distance_per_truck < 450, 'WARNING: Cost function for the annual capex of the cH2 tube trailer out of range! Annual distance per truck is {:.2f} km'.format(annual_distance_per_truck)

        annual_fleet_costs = pyo.value(model.edges_ch2_tube_trailer_cost_04)  # [tNOK2024/a]
        annual_fleet_costs_nonlinear = pyo.value(get_fleet_costs_nonlinear_cH2_tube_trailer(model, total_driven_distance / 1000))

        result_string += '\tFleet operation: {:.2f} [Mkm] using {:.0f} trucks on {:.0f} trips with an average one-way distance of {:.0f} [km]\n'.format(total_driven_distance/1000/1000, num_trucks_in_fleet, total_num_trips, total_driven_distance/total_num_trips/2)
        result_string += '\tFleet operation: {:.2f} [tNOK2024/a] annual costs per truck and {:.2f} [NOK2024/kg H2] average transport costs (linear cost used in model)\n'.format(annual_fleet_costs / num_trucks_in_fleet, annual_fleet_costs * 1000 / (annual_transported_ch2 * 1000 * 1000))
        result_string += '\tFleet operation: {:.2f} [tNOK2024/a] annual costs per truck and {:.2f} [NOK2024/kg H2] average transport costs (nonlinear cost estimate)\n'.format(annual_fleet_costs_nonlinear / num_trucks_in_fleet, annual_fleet_costs_nonlinear * 1000 / (annual_transported_ch2 * 1000 * 1000))
    #######################################################################################################

    #######################################################################################################
    # cH2 Shipping terminals (export)

    result_string += 'cH2 shipping terminals (export):\n'

    for ch2_shipping_terminal in developed_cH2_shipping_terminals_export.index.to_list():
        expansion_size = pyo.value(model.nodes_ch2_shipping_terminal_expansion_size_05[ch2_shipping_terminal]) # [kt hydrogen per year]
        ch2_tube_trailer_terminal_size = pyo.value(model.nodes_ch2_shipping_terminal_ch2_tube_trailer_expansion_size_05[ch2_shipping_terminal])  # [kt hydrogen per year]
        production_cost_absolute = pyo.value(model.nodes_ch2_shipping_terminal_cost_05[ch2_shipping_terminal]) # [tNOK2024 per year]
        production_cost_specific = production_cost_absolute / expansion_size / 1000 # [tNOK2024 per year] / [kt hydrogen per year] / 1000 = [NOK2024 / kg hydrogen]
        number_of_outgoing_trips = sum(pyo.value(model.edges_ch2_shipping_num_trips_06[ch2_shipping_terminal, ch2_shipping_terminal_import]) for ch2_shipping_terminal_import in developed_cH2_shipping_terminals_import.index.to_list())
        utilization = number_of_outgoing_trips * (metadata['cH2 Shipping Terminal (export) Filling time [min]']) / 60 # [hours per year]

        total_shipping_terminal_costs_export += production_cost_absolute

        # Limits for the terminal throughput
        # At theoretical maximum utilization (at every moment in time, a ship is loading), the ship loading power is equal to terminal throughput

        ship_payload = metadata['cH2 Ship Payload per module [kgH2]'] * metadata['cH2 Ship Num Modules [-]']  # [kg hydrogen per trip]
        ship_loading_time = metadata['cH2 Shipping Terminal (export) Filling time [min]']
        ship_loading_power = ship_payload / (ship_loading_time/60) # [kg/h hydrogen]
        ship_loading_power = ship_loading_power / 1501 * 50 # [MW hydrogen]

        expansion_size_min = metadata['cH2 Shipping Terminal (export) SIZE - min [MW]'] / 100 * 26.2975
        expansion_size_max = ship_loading_power / 100 * 26.2975

        if plot_nodes:
            tooltip_html = '''<b>cH2 Shipping terminal</b><br>
                    Index: {}<br>
                    Production level: {:.2f} [kt hydrogen/a]<br>
                    Production limit: {:.2f} [kt hydrogen/a]<br>
                    Production costs: {:.2f} [NOK/kg]'''.format(ch2_shipping_terminal, expansion_size, expansion_size_max, production_cost_specific)

            folium.Marker(location=tuple(reversed(developed_cH2_shipping_terminals_export.loc[ch2_shipping_terminal, 'geometry_4326'])), icon=folium.Icon(color='black', icon_color='white', icon='ship', prefix='fa'), tooltip=tooltip_html).add_to(map_level_2)

        result_string += ('\tcH2 Shipping Terminal {}: {:.2f} / {:.2f} / {:.2f} [kt hydrogen/a] @ {:.2f} [NOK/kg hydrogen]  || cH2 Tube Trailer Terminal Size: {:.2f} kt/a || Utilization: {:0f} h/a\n'
                          .format(ch2_shipping_terminal, expansion_size_min, expansion_size, expansion_size_max, production_cost_specific, ch2_tube_trailer_terminal_size, utilization))
    #######################################################################################################

    #######################################################################################################
    # Wood chip shipping terminals (export)

    result_string += 'Wood chip shipping terminals (export):\n'

    for timber_shipping_terminal in developed_wood_chip_shipping_terminals_export.index.to_list():
        expansion_size = pyo.value(model.nodes_wood_chip_shipping_terminal_expansion_size_05[timber_shipping_terminal])  # [kt timber per year]
        production_cost_absolute = pyo.value(model.nodes_wood_chip_shipping_terminal_cost_05[timber_shipping_terminal])  # [tNOK2024 per year]
        production_cost_specific = production_cost_absolute / expansion_size  # [tNOK2024 per year] / [kt timber per year] / 1000 = [NOK2024 / t timber]
        number_of_outgoing_trips = sum(pyo.value(model.edges_wood_chip_shipping_num_trips_06[timber_shipping_terminal, wc_shipping_terminal_import]) for wc_shipping_terminal_import in developed_wood_chip_shipping_terminals_import.index.to_list())
        utilization = number_of_outgoing_trips * metadata['Timber ship transfer time export terminal [h]']

        total_shipping_terminal_costs_export += production_cost_absolute

        if plot_nodes:
            tooltip_html = '''<b>Timber Shipping terminal</b><br>
                    Index: {}<br>
                    Production level: {:.2f} [kt timber/a]<br>
                    Production costs: {:.2f} [NOK/t]'''.format(timber_shipping_terminal, expansion_size, production_cost_specific)

            folium.Marker(location=tuple(reversed(developed_wood_chip_shipping_terminals_export.loc[timber_shipping_terminal, 'geometry_4326'])), icon=folium.Icon(color='black', icon_color='white', icon='ship', prefix='fa'), tooltip=tooltip_html).add_to(map_level_2)

        result_string += ('\tWood chip shipping Terminal {}: {:.2f} [kt timber/a] @ {:.2f} [NOK/t timber] || Utilization: {:0f} h/a\n'
                          .format(timber_shipping_terminal, expansion_size, production_cost_specific, utilization))
    #######################################################################################################

    #######################################################################################################
    # cH2 Shipping_Routes

    developed_ch2_shipping_routes = [(ch2_shipping_terminal_export, ch2_shipping_terminal_import) for ch2_shipping_terminal_export in model.cH2_shipping_terminal_indices_05 for ch2_shipping_terminal_import in model.cH2_shipping_terminal_indices_07 if
                                         pyo.value(model.edges_ch2_shipping_amount_06[ch2_shipping_terminal_export, ch2_shipping_terminal_import]) > 0.01]

    # Fleet costs
    fleet_fixed_costs = pyo.value(model.edges_ch2_shipping_fleet_cost_06)  # [tNOK2024/a]
    fleet_variable_costs = sum(pyo.value(model.edges_ch2_shipping_cost_06[export_terminal, import_terminal]) for export_terminal, import_terminal in developed_ch2_shipping_routes)  # [tNOK2024/a]
    total_shipping_costs += fleet_fixed_costs + fleet_variable_costs

    fleet_transported_ch2 = sum(pyo.value(model.edges_ch2_shipping_amount_06[export_terminal, import_terminal]) for export_terminal, import_terminal in developed_ch2_shipping_routes)  # [kt hydrogen per year]
    required_number_of_ships_actual = 0

    result_string += 'cH2 Shipping Routes:\n'
    total_number_of_trips = 0
    total_distance_sailed = 0

    developed_ch2_shipping_routes_names = []

    for route in developed_ch2_shipping_routes:
        edge_thickness = pyo.value(model.edges_ch2_shipping_amount_06[route[0], route[1]]) # [kt hydrogen per year]
        distance = edge_data_shipping_distance_06.loc[route[0], route[1]] # One-way distance [km]

        # Translate index into name
        developed_ch2_shipping_routes_names.append((route[0], route[1]))

        # Number of ships
        num_trips = pyo.value(model.edges_ch2_shipping_num_trips_06[route[0], route[1]])  # [-]
        num_ships_required = get_required_number_of_ships_on_route(route[0], route[1], edge_thickness, data)

        required_number_of_ships_actual += num_ships_required
        total_number_of_trips += num_trips
        total_distance_sailed += distance * num_trips * 2

        specific_shipping_cost = pyo.value(model.edges_ch2_shipping_cost_06[route[0], route[1]]) / edge_thickness / 1000 # [tNOK2024/a] / [kt/a] * 1000 = [NOK2024/kg]

        result_string += '\tcH2 shipping from {} to {}: {:.2f} [kt/a] for {:.2f} [km] @ {:.2f} [NOK/kg H2] variable costs requiring {:.2f} ships and {:.0f} trips\n'.format(route[0], route[1], edge_thickness, distance,
                                                                                                                                                                           specific_shipping_cost, num_ships_required, num_trips)

    if fleet_transported_ch2 > 0:
        result_string += '\tFleet number of ships: {:.2f} required (actually) -> {:.0f} used\n'.format(required_number_of_ships_actual, pyo.value(model.edges_ch2_shipping_fleet_num_ships_06))
        result_string += '\tFleet operation: {:.2f} [Mkm] on {:.0f} trips with an average one-way distance of {:.0f} [km], transporting {:.2f} [kt/a] cH2\n'.format(total_distance_sailed/1000/1000, total_number_of_trips, total_distance_sailed/total_number_of_trips/2, final_hydrogen_demand_kta)
        result_string += '\tFleet costs: {:.2f} [NOK/kg H2] fixed || {:.2f} [NOK/kg H2] variable\n'.format(fleet_fixed_costs * 1000 / (fleet_transported_ch2 * 1000 * 1000), fleet_variable_costs * 1000 / (fleet_transported_ch2 * 1000 * 1000))
    #######################################################################################################

    #######################################################################################################
    # Wood chip shipping routes

    developed_wood_chip_shipping_routes = [(terminal_export, terminal_import) for terminal_export in model.wc_shipping_terminal_indices_05 for terminal_import in model.wc_shipping_terminal_indices_07 if
                                     pyo.value(model.edges_wood_chip_shipping_amount_06[terminal_export, terminal_import]) > 0.01]

    result_string += 'Wood chip shipping routes:\n'
    total_number_of_trips = 0
    total_distance_sailed = 0

    # Fleet costs
    fleet_fixed_costs = pyo.value(model.edges_wood_chip_shipping_fleet_cost_06) # [tNOK2024/a]
    fleet_variable_costs = sum(pyo.value(model.edges_wood_chip_shipping_cost_06[export_terminal, import_terminal]) for export_terminal, import_terminal in developed_wood_chip_shipping_routes) # [tNOK2024/a]
    fleet_transported_timber = sum(pyo.value(model.edges_wood_chip_shipping_amount_06[export_terminal, import_terminal]) for export_terminal, import_terminal in developed_wood_chip_shipping_routes) # [kt timber per year]
    fleet_transported_equivalent_h2 = fleet_transported_timber / metadata['Biomass Gasification OPEX Timber consumption [kg/kgH2]'] # [kt timber per year]

    total_shipping_costs += fleet_fixed_costs + fleet_variable_costs

    # Fleet ships
    required_number_of_ships = 0  # Minimum number of ships in the fleet [-]

    # Routes
    for route in developed_wood_chip_shipping_routes:
        edge_thickness = pyo.value(model.edges_wood_chip_shipping_amount_06[route[0], route[1]]) # [kt timber per year]
        num_trips = pyo.value(model.edges_wood_chip_shipping_num_trips_06[route[0], route[1]]) # [#trips per year]
        distance = edge_data_shipping_distance_06.loc[route[0], route[1]] # One-way distance [km]
        specific_shipping_cost_timber = pyo.value(model.edges_wood_chip_shipping_cost_06[route[0], route[1]]) / edge_thickness # Variable shipping cost [tNOK2024/a] / [kt/a] = [NOK2024/t timber]

        shipping_amount = pyo.value(model.edges_wood_chip_shipping_amount_06[route[0], route[1]])  # [kt timber per year]
        number_of_ships_on_route = get_required_number_of_ships_on_route_timber(route[0], route[1], shipping_amount, data)  # float, number of ships on route A-B [-]

        required_number_of_ships += number_of_ships_on_route
        total_number_of_trips += num_trips
        total_distance_sailed += distance * num_trips * 2

        result_string += '\tWood chip shipping from {} to {}: {:.2f} [kt/a] for {:.2f} [km] @ {:.2f} [NOK/t timber] variable costs on {:.0f} trips\n'.format(route[0], route[1], edge_thickness, distance, specific_shipping_cost_timber, num_trips)

    if fleet_transported_timber > 0:
        result_string += '\tFleet number of ships: {:.2f} required -> {:.0f} used\n'.format(required_number_of_ships, pyo.value(model.edges_wood_chip_shipping_fleet_num_ships_06))
        result_string += '\tFleet operation: {:.2f} [Mkm] on {:.0f} trips with an average one-way distance of {:.0f} [km], transporting {:.2f} [Mt/a] wood chips\n'.format(total_distance_sailed/1000/1000, total_number_of_trips, total_distance_sailed/total_number_of_trips/2, fleet_transported_timber/1000)
        result_string += '\tFleet costs (actual): {:.2f} [NOK/t timber] fixed || {:.2f} [NOK/t timber] variable\n'.format(fleet_fixed_costs * 1000 / (fleet_transported_timber*1000), fleet_variable_costs * 1000 / (fleet_transported_timber*1000))
        result_string += '\tFleet costs (H2 equivalent): {:.2f} [NOK/kg H2] fixed || {:.2f} [NOK/kg H2] variable\n'.format(fleet_fixed_costs * 1000 / (fleet_transported_equivalent_h2*1000*1000), fleet_variable_costs * 1000 / (fleet_transported_equivalent_h2*1000*1000))
    #######################################################################################################

    #######################################################################################################
    # cH2 shipping terminals (import)

    result_string += 'cH2 shipping terminals (import):\n'

    for ch2_shipping_terminal in developed_cH2_shipping_terminals_import.index.to_list():
        expansion_size = pyo.value(model.nodes_ch2_shipping_terminal_expansion_size_07[ch2_shipping_terminal])  # [kt hydrogen per year]
        production_cost_absolute = pyo.value(model.nodes_ch2_shipping_terminal_cost_07[ch2_shipping_terminal])  # [tNOK2024 per year]
        production_cost_specific = production_cost_absolute / expansion_size / 1000  # [tNOK2024 per year] / [kt hydrogen per year] / 1000 = [NOK2024 / kg hydrogen]
        number_of_incoming_trips = sum(pyo.value(model.edges_ch2_shipping_num_trips_06[ch2_shipping_terminal_export, ch2_shipping_terminal]) for ch2_shipping_terminal_export in developed_cH2_shipping_terminals_export.index.to_list())
        utilization = number_of_incoming_trips * metadata['cH2 Shipping Terminal (import) Emptying time [min]'] / 60  # [hours per year]

        total_shipping_terminal_costs_import += production_cost_absolute

        if plot_nodes:
            tooltip_html = '''<b>cH2 Shipping terminal</b><br>
                        Index: {}<br>
                        Production level: {:.2f} [kt hydrogen/a]<br>
                        Production costs: {:.2f} [NOK/kg]'''.format(ch2_shipping_terminal, expansion_size, production_cost_specific)

            folium.Marker(location=tuple(reversed(developed_cH2_shipping_terminals_import.loc[ch2_shipping_terminal, 'geometry_4326'])), icon=folium.Icon(color='black', icon_color='white', icon='ship', prefix='fa'), tooltip=tooltip_html).add_to(map_level_2)

        result_string += ('\tcH2 Shipping Terminal {}: {:.2f} [kt hydrogen/a] @ {:.2f} [NOK/kg hydrogen] || Utilization: {:.0f} h/a\n'
                          .format(ch2_shipping_terminal, expansion_size, production_cost_specific, utilization))
    #######################################################################################################

    #######################################################################################################
    # Wood chip shipping terminals (import)

    result_string += 'Timber shipping terminals (import):\n'

    for terminal in developed_wood_chip_shipping_terminals_import.index.to_list():
        expansion_size_timber = sum(pyo.value(model.edges_wood_chip_shipping_amount_06[export_terminal, terminal]) for export_terminal in model.wc_shipping_terminal_indices_05) # [kt timber per year]
        expansion_size_hydrogen = expansion_size_timber / metadata['Biomass Gasification OPEX Timber consumption [kg/kgH2]'] # Equivalent size [kt hydrogen per year]
        num_incoming_trips = sum(pyo.value(model.edges_wood_chip_shipping_num_trips_06[export_terminal, terminal]) for export_terminal in model.wc_shipping_terminal_indices_05)  # [#trips per year]
        utilization = num_incoming_trips * metadata['Timber ship transfer time import terminal [h]']

        production_cost_absolute = pyo.value(model.nodes_wood_chip_shipping_terminal_cost_07[terminal]) # [tNOK2024 per year]
        production_cost_specific = production_cost_absolute / expansion_size_hydrogen / 1000 # [tNOK2024 per year] / [kt hydrogen per year] / 1000 = [NOK2024 / kg hydrogen]

        total_shipping_terminal_costs_import += production_cost_absolute

        if plot_nodes:
            tooltip_html = '''<b>Timber Shipping terminal</b><br>
                        Index: {}<br>
                        Terminal size: {:.2f} [kt timber/a]<br>
                        Production costs: {:.2f} [NOK/kg H2]'''.format(terminal, expansion_size_timber, production_cost_specific)

            folium.Marker(location=tuple(reversed(developed_wood_chip_shipping_terminals_import.loc[terminal, 'geometry_4326'])), icon=folium.Icon(color='black', icon_color='white', icon='ship', prefix='fa'), tooltip=tooltip_html).add_to(map_level_2)

        result_string += ('\tTimber Shipping Terminal {}: {:.2f} [kt timber/a] receiving {:.0f} trips @ {:.2f} [NOK/kg hydrogen] || Utilization: {:0f} h/a\n'
                          .format(terminal, expansion_size_timber, num_incoming_trips, production_cost_specific, utilization))
    #######################################################################################################

    #######################################################################################################
    # Biomass Gasification Plant (DE)

    result_string += 'Biomass Gasification Plant (DE):\n'

    for terminal in developed_biomass_gasification_hubs_07:
        expansion_size_hydrogen = pyo.value(model.nodes_biomass_gasification_hub_expansion_size_07[terminal]) # [kt hydrogen per year]
        timber_supply = sum([pyo.value(model.edges_wood_chip_shipping_amount_06[export_terminal, terminal]) for export_terminal in model.wc_shipping_terminal_indices_05]) # [kt hydrogen per year]

        production_cost_absolute = pyo.value(model.nodes_biomass_gasification_hub_cost_07[terminal]) # [tNOK2024 per year]
        production_cost_specific = production_cost_absolute / expansion_size_hydrogen / 1000 # [tNOK2024 per year] / [kt hydrogen per year] / 1000 = [NOK2024 / kg hydrogen]

        total_gasification_costs += production_cost_absolute

        result_string += ('\tGasification Plant {}: {:.2f} [kt timber/a] -> {:.2f} [kt hydrogen/a] @ {:.2f} [NOK/kg hydrogen]\n'
                          .format(terminal, timber_supply, expansion_size_hydrogen, production_cost_specific))
    #######################################################################################################

    #######################################################################################################
    # Bar chart with total cost distribution

    cost_categories = ['Timber Production', 'Gasification', 'Timber Truck', 'Tube Trailer', 'Shipping Terminals (export)', 'Shipping Terminals (import)', 'Shipping']
    cost_values_absolute = np.array([total_timber_production_costs, total_gasification_costs, total_timber_truck_costs, total_tube_trailer_costs, total_shipping_terminal_costs_export, total_shipping_terminal_costs_import, total_shipping_costs])  # [tNOK2024/a]
    cost_values_specific = cost_values_absolute * 1000 / (final_hydrogen_demand_kta * 1000 * 1000)  # [NOK/kg H2]

    # Create bar chart
    if generate_bar_charts:
        fig, ax = plt.subplots(figsize=(9 * 0.7, 6 * 0.7))
        plt.bar(cost_categories, cost_values_specific, color='skyblue')
        plt.title('Breakdown of levelized costs of hydrogen')
        plt.ylabel('Cost [NOK/kg H2]')
        plt.grid(axis='y', linestyle='--', alpha=0.7)
        plt.xticks(rotation=45, ha='right')
        plt.tight_layout()
        plt.savefig('plots/lcoh.png', dpi=600)
        plt.show()

    #######################################################################################################

    #######################################################################################################
    # Bar chart with total emissions distribution

    emission_categories = ['Timber Production', 'Gasification', 'Timber Truck', 'Tube Trailer', 'Shipping Terminals', 'Shipping']
    emission_values_absolute = np.array([pyo.value(model.nodes_biomass_production_site_emission_01),
                                         pyo.value(model.nodes_biomass_gasification_hub_emission_03) + pyo.value(model.nodes_biomass_gasification_hub_emission_07),
                                         pyo.value(model.edges_timber_truck_emissions),
                                         pyo.value(model.edges_ch2_tube_trailer_emissions_04),
                                         pyo.value(model.nodes_ch2_shipping_terminal_emission_05) + pyo.value(model.nodes_ch2_shipping_terminal_emission_07) + pyo.value(model.nodes_wood_chip_shipping_terminal_emission_05),
                                         pyo.value(model.edges_ch2_shipping_fleet_emissions_06) + pyo.value(model.edges_wood_chip_shipping_fleet_emissions_06)])  # [tCO2eq/a]
    emission_values_specific = emission_values_absolute * 1000 * 1000 / (final_hydrogen_demand_kta * 1000 * 1000)  # [gCO2eq/kg H2]

    # Create bar chart
    if generate_bar_charts:
        fig, ax = plt.subplots(figsize=(9 * 0.7, 6 * 0.7))
        plt.bar(emission_categories, emission_values_specific, color='skyblue')
        plt.title('Breakdown of operational supply chain emissions')
        plt.ylabel('Emissions [gCO2eq/kg H2]')
        plt.grid(axis='y', linestyle='--', alpha=0.7)
        plt.xticks(rotation=45, ha='right')
        plt.tight_layout()
        plt.show()
    #######################################################################################################

    #######################################################################################################
    # Plot of map with supply chain

    if plot_nodes:
        marker_size_sc_lv1 = 18
        marker_size_sc_lv2 = 92
        marker_size_sc_lv3 = 62

        fig_sc, ax_sc = plt.subplots(figsize=(10, 18), constrained_layout=True)

        # Forest resources
        gdf_plot = node_data_biomass_01.loc[developed_timber_production_sites, :]
        ax_sc = gdf_plot.plot(ax=ax_sc, color='black', markersize=marker_size_sc_lv1, marker='.', label='Forest resources', zorder=1)

        # Timber transport
        if len(developed_timber_transport_routes_02) > 0:
            gdf_plot = data['edge_data_timber_trucking_route_02'].loc[developed_timber_transport_routes_02, :]
            ax_sc = gdf_plot.plot(ax=ax_sc, color='black', edgecolor='black', linewidth=0.5, label='Timber Truck', zorder=1)

        if len(developed_timber_transport_routes_04) > 0:
            gdf_plot = data['edge_data_timber_trucking_route_04'].loc[developed_timber_transport_routes_04, :]
            ax_sc = gdf_plot.plot(ax=ax_sc, color='black', edgecolor='black', linewidth=0.5, label='Timber Truck', zorder=1)

        # Biomass gasification (NO)
        if len(developed_biomass_gasification_hubs_03) > 0:
            ax_sc = developed_biomass_gasification_hubs_03.plot(ax=ax_sc, color='navy', markersize=marker_size_sc_lv3, marker='p', label='H2 Production', zorder=3)

        # cH2 Tube Trailer
        if len(developed_ch2_tube_trailer_routes) > 0:
            gdf_plot = data['edge_data_cH2_trucking_route_04'].loc[developed_ch2_tube_trailer_routes, :]
            ax_sc = gdf_plot.plot(ax=ax_sc, color='navy', edgecolor='navy', linewidth=0.5, label='cH2 Tube Trailer', zorder=1)

        # Shipping terminals (export)
        if len(developed_cH2_shipping_terminals_export) > 0:
            ax_sc = developed_cH2_shipping_terminals_export.plot(ax=ax_sc, color='#a02b92', markersize=marker_size_sc_lv2, marker='8', label='cH2 Shipping Terminal', zorder=2)

        if len(developed_wood_chip_shipping_terminals_export) > 0:
            ax_sc = developed_wood_chip_shipping_terminals_export.plot(ax=ax_sc, color='#a02b92', markersize=marker_size_sc_lv2, marker='8', label='Wood Shipping Terminal', zorder=2)

        # Shipping routes
        if len(developed_ch2_shipping_routes) > 0:
            gdf_plot = data['edge_data_shipping_paths_06'].loc[developed_ch2_shipping_routes_names, :]
            ax_sc = gdf_plot.plot(ax=ax_sc, color='black', edgecolor='black', linewidth=1, linestyle='dashed', label='cH2 Shipping Route', zorder=1)

        if len(developed_wood_chip_shipping_routes) > 0:
            gdf_plot = data['edge_data_shipping_paths_06'].loc[developed_wood_chip_shipping_routes, :]
            ax_sc = gdf_plot.plot(ax=ax_sc, color='black', edgecolor='black', linewidth=1, linestyle='dashed', label='Wood Shipping Route', zorder=1)

        # Shipping terminals (import)
        if len(developed_cH2_shipping_terminals_import) > 0:
            ax_sc = developed_cH2_shipping_terminals_import.plot(ax=ax_sc, color='#a02b92', markersize=marker_size_sc_lv2, marker='8', zorder=2)

        if len(developed_wood_chip_shipping_terminals_import) > 0:
            ax_sc = developed_wood_chip_shipping_terminals_import.plot(ax=ax_sc, color='#a02b92', markersize=marker_size_sc_lv2, marker='8', zorder=2)

        # Biomass gasification (DE)
        if len(developed_biomass_gasification_hubs_07) > 0:
            ax_sc = developed_wood_chip_shipping_terminals_import.plot(ax=ax_sc, color='navy', markersize=marker_size_sc_lv3, marker='*', label='H2 Production', zorder=3)

        # Basemap
        ax_sc.set_xlim(-168837.52327370193, 820610.8733992758)
        ax_sc.set_ylim(5861332.626080652, 7885175.056468202)
        cx.add_basemap(ax_sc, crs='EPSG:25833', source=cx.providers.Esri.WorldTerrain, attribution=False)

        # Remove axis ticks
        ax_sc.set_xticks([])
        ax_sc.set_yticks([])

        # Text annotation
        ax_sc.text(
            0.95, 0.035, case_name,
            transform=ax_sc.transAxes,
            ha='right', va='bottom',
            fontweight='bold',
            fontsize=18
        )

        ax_sc.text(
            0.95, 0.00, '{:.1f} [NOK/kgH$_2$] for {:.0f} MW\n'.format(pyo.value(model.total_cost(model)) * 1000 / (final_hydrogen_demand_kta * 1000 * 1000), final_hydrogen_demand_kta * 100 / 26.2975),
            transform=ax_sc.transAxes,
            ha='right', va='bottom',
            fontsize=18
        )

        plt.legend(loc='upper left', fontsize=18)
        # plt.tight_layout()
        plt.savefig('plots/map_supply_chain.png', dpi=800)
        plt.show()

    #######################################################################################################

    #######################################################################################################
    # General analysis
    objective_value = pyo.value(model.total_cost(model))

    result_string += '\nSUM: Totally consumed timber: {:.2f} [kt/a]\n'.format(sum(pyo.value(model.nodes_biomass_production_site_expansion_size_01[idx]) for idx in model.timber_resources_indices_01))
    result_string += 'SUM: Approximate economic timber potential: {:.2f} [kt/a]\n'.format(sum(node_data_biomass_01.loc[:, 'Potential [kt/a]']))

    result_string += '\nValue of objective function: {:.2f} [MNOK2024/a]\n'.format(objective_value / 1000)
    result_string += 'Value of objective function: {:.2f} [NOK/kgH2] for {:.2f} [kt/a]\n'.format(objective_value * 1000 / (final_hydrogen_demand_kta * 1000 * 1000), final_hydrogen_demand_kta)
    result_string += 'Specific emissions: {:.2f} [gCO2eq/kg H2]\n'.format(sum(emission_values_specific))

    # Print node variables
    if pprint_variables:
        print('\n### NODE VARIABLES ### \n')
        variables = [model.nodes_biomass_gasification_hub_expansion_size_03, model.nodes_ch2_shipping_terminal_expansion_size_05, model.nodes_wood_chip_shipping_terminal_expansion_size_05, model.nodes_ch2_shipping_terminal_expansion_size_07, model.nodes_biomass_gasification_hub_expansion_size_07]
        for variable in variables:
            variable.pprint()
        print('\n')

    # Sanity check
    assert abs(sum(cost_values_absolute) - objective_value) < 10, 'Either error in objective function or analysis'

    #######################################################################################################

    #######################################################################################################
    # Export

    if plot_nodes:
        map_level_1.save('plots/' + 'SelectedNodes_Level_1_' + RUN_ID + ".html")
        webbrowser.open_new_tab(os.getcwd() + '/plots/' + 'SelectedNodes_Level_1_' + RUN_ID + ".html")

        map_level_2.save('plots/' + 'SelectedNodes_Level_2_' + RUN_ID + ".html")
        webbrowser.open_new_tab(os.getcwd() + '/plots/' + 'SelectedNodes_Level_2_' + RUN_ID + ".html")

    if write_results:
        filename = 'results/SOLVER_OUTPUT_' + RUN_ID + '.txt'

        with open(filename, "w") as file:
            file.write(result_string)
            file.close()

    if print_results:
        print('\n\n')
        print(result_string)

    return objective_value, cost_values_specific, emission_values_specific