#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Create nodes where shipping terminals could be built
#################################################################################################################################################################################################################################################


################################################################################################################################
# IMPORT
import geopandas as gpd
import time
import os
import folium
import webbrowser
import pandas as pd
pd.options.mode.chained_assignment = None

start_time = time.time()
################################################################################################################################


################################################################################################################################
print('Importing data')

assert os.path.exists('../X_GLOBAL/data_processed/dataset_tettsteder.feather'), 'Process raw data first!'

# Import datasets
gdf_tettsteder = gpd.read_feather('../X_GLOBAL/data_processed/dataset_tettsteder.feather')
gdf_coastline = gpd.read_feather('../X_GLOBAL/data_processed/dataset_coastline.feather')
electricity_price = pd.read_excel('../X_GLOBAL/data_processed/dataset_landsdeler.xlsx', sheet_name='electricity_price', index_col=0).loc[:,['Sum [NOK2024/MWh]']]

# Format index (Electricity Day Ahead NO1 -> NO1)
assert electricity_price.index.values[0][-3:-1] == 'NO', 'Check validity of electricity price index!'
electricity_price.index = electricity_price.index.str.split(' ').str[-1]

print('Time passed: {:.2f} seconds'.format(time.time() - start_time))
################################################################################################################################


################################################################################################################################

################################################################################################
# Select tettsteder that could be potential places for building shipping terminals

gdf_shipping_terminals_all = gpd.sjoin_nearest(gdf_tettsteder.copy(), gdf_coastline, how='left', distance_col='distance_to_coast')
gdf_shipping_terminals_all = gdf_tettsteder[gdf_shipping_terminals_all['distance_to_coast'] < 2000]

# Select 20 of these places, spread out evenly across the map
# Method:
# 1. Select the first hub as a start hub
# 2. Calculate the score of all other hubs based on their location in respect to the selected hubs
#   a) The further away a hub is, the lower the score
#   b) The hub with the lowest score gets selected and is added to the selected hubs

gdf_shipping_terminals = gdf_shipping_terminals_all.copy(deep=True)
num_desired_hubs = 20

# Start with the first hub
start_hub = gdf_shipping_terminals.iloc[0]
selected_hubs = [start_hub['geometry']]
selected_hubs_idx = [start_hub.name]
gdf_shipping_terminals.drop(start_hub.name, axis='index', inplace=True)

while len(selected_hubs) < num_desired_hubs and len(gdf_shipping_terminals) > 1:
    gdf_shipping_terminals['score'] = 0

    for hub in selected_hubs:
        gdf_shipping_terminals['score'] = gdf_shipping_terminals['score'] + 1/ (gdf_shipping_terminals['geometry'].distance(hub) * gdf_shipping_terminals['geometry'].distance(hub))

    gdf_shipping_terminals.sort_values('score', ascending=True, inplace=True)

    new_hub = gdf_shipping_terminals.iloc[0]
    selected_hubs.append(new_hub['geometry'])
    selected_hubs_idx.append(new_hub.name)
    gdf_shipping_terminals.drop(new_hub.name, axis='index', inplace=True)

gdf_shipping_terminals = gdf_shipping_terminals_all.loc[selected_hubs_idx,:] # Select hubs

################################################################################################

################################################################################################
# Append cost factors

_ = gdf_shipping_terminals[['ElSpotOmr']].map(lambda e:electricity_price.loc[e,'Sum [NOK2024/MWh]']).rename(columns={'ElSpotOmr': 'Electricity price [NOK2024/MWh]'}) # OPEX is different in each county (= electricity price in that county)
gdf_shipping_terminals = pd.concat([gdf_shipping_terminals, _], axis=1)

gdf_shipping_terminals.rename(columns={'centroid_4326': 'geometry_4326'}, inplace=True)
gdf_shipping_terminals['tettstedsn'] = gdf_shipping_terminals['tettstedsn'].str.split('/').str[0]
gdf_shipping_terminals.index = gdf_shipping_terminals['tettstedsn'].values
gdf_shipping_terminals.drop('tettstedsn', axis='columns', inplace=True)

################################################################################################

################################################################################################
# Plot selected terminals
m = folium.Map(tiles="cartodb positron")
_ = gdf_shipping_terminals.apply(lambda row: folium.Marker(location=tuple(reversed(row.geometry_4326)), icon=folium.Icon(color='black', icon_color='black'), tooltip=row.name).add_to(m), axis=1)

plot_name = 'possible_cH2_shipping_terminals'
m.save('plots/' + plot_name + ".html")
webbrowser.open_new_tab(os.getcwd() + '/plots/' + plot_name + ".html")

# Export
gdf_shipping_terminals.to_feather('../nodes_tertiary_cH2_shipping_terminal_export/output_data/node_data_05_cH2_shipping_terminals_export.feather')
################################################################################################

################################################################################################################################