#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Create nodes where gasification plants could be built in Nowary
#################################################################################################################################################################################################################################################


################################################################################################################################
# IMPORT
import geopandas as gpd
import time
import os
import folium
import webbrowser
import pandas as pd
pd.options.mode.chained_assignment = None

start_time = time.time()
################################################################################################################################


################################################################################################################################
print('Importing data')

assert os.path.exists('../X_GLOBAL/data_processed/dataset_tettsteder.feather'), 'Process raw data first!'

# Import datasets
gdf_tettsteder = gpd.read_feather('../X_GLOBAL/data_processed/dataset_tettsteder.feather')
electricity_price = pd.read_excel('../X_GLOBAL/data_processed/dataset_landsdeler.xlsx', sheet_name='electricity_price', index_col=0).loc[:,['Sum [NOK2024/MWh]']]
timber_price = pd.read_excel('../X_GLOBAL/data_processed/dataset_landsdeler.xlsx', sheet_name='skog_data', index_col=0).loc[:,['Timber price [NOK2023/m3]']]

# Format index (Electricity Day Ahead NO1 -> NO1)
assert electricity_price.index.values[0][-3:-1] == 'NO', 'Check validity of electricity price index!'
electricity_price.index = electricity_price.index.str.split(' ').str[-1]

print('Time passed: {:.2f} seconds'.format(time.time() - start_time))
################################################################################################################################


################################################################################################################################

################################################################################################

# Select 40 of these places, spread out evenly across the map
# Method:
# 1. Select the first hub as a start hub
# 2. Calculate the score of all other hubs based on their location in respect to the selected hubs
#   a) The further away a hub is, the lower the score
#   b) The hub with the lowest score gets selected and is added to the selected hubs

gdf_biomass_hubs = gdf_tettsteder.copy(deep=True)
num_desired_hubs = 40

# Start with the first hub
start_hub = gdf_biomass_hubs.iloc[0]
selected_hubs = [start_hub['geometry']]
selected_hubs_idx = [start_hub.name]
gdf_biomass_hubs.drop(start_hub.name, axis='index', inplace=True)

while len(selected_hubs) < num_desired_hubs and len(gdf_biomass_hubs) > 1:
    gdf_biomass_hubs['score'] = 0

    for hub in selected_hubs:
        gdf_biomass_hubs['score'] = gdf_biomass_hubs['score'] + 1/ (gdf_biomass_hubs['geometry'].distance(hub) * gdf_biomass_hubs['geometry'].distance(hub))

    gdf_biomass_hubs.sort_values('score', ascending=True, inplace=True)

    new_hub = gdf_biomass_hubs.iloc[0]
    selected_hubs.append(new_hub['geometry'])
    selected_hubs_idx.append(new_hub.name)
    gdf_biomass_hubs.drop(new_hub.name, axis='index', inplace=True)

gdf_biomass_hubs = gdf_tettsteder.loc[selected_hubs_idx,:] # Select hubs
gdf_biomass_hubs.index = ['biomass_gasification_plant_' + str(el) for el in gdf_biomass_hubs.index]

################################################################################################

################################################################################################
# Calculate specific cost data

gdf_biomass_hubs = pd.concat([gdf_biomass_hubs, gdf_biomass_hubs[['ElSpotOmr']].map(lambda e:electricity_price.loc[e,'Sum [NOK2024/MWh]']).rename(columns={'ElSpotOmr': 'Electricity price [NOK2024/MWh]'})], axis='columns')
gdf_biomass_hubs = pd.concat([gdf_biomass_hubs, gdf_biomass_hubs[['landsdel']].map(lambda e:timber_price.loc[e,'Timber price [NOK2023/m3]']).rename(columns={'landsdel': 'Timber price [NOK2023/m3]'})], axis='columns')

################################################################################################

################################################################################################
# Plot selected biomass hubs
m = folium.Map(tiles="cartodb positron")
_ = gdf_biomass_hubs.apply(lambda row: folium.Marker(location=tuple(reversed(row.centroid_4326)), icon=folium.Icon(color='black', icon_color='black'), tooltip=row.tettstedsn).add_to(m), axis=1)

plot_name = 'possible_biomass_gasification_plants'
m.save('plots/' + plot_name + ".html")
webbrowser.open_new_tab(os.getcwd() + '/plots/' + plot_name + ".html")

# Export
gdf_biomass_hubs.drop(['centroid_4326' ], axis='columns', inplace=True)
gdf_biomass_hubs.to_feather('../nodes_secondary_biomass_gasification/output_data/node_data_03_biomass_gasification.feather')
################################################################################################

################################################################################################################################