#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Read and process data
#################################################################################################################################################################################################################################################


################################################################################################################################
# IMPORT
import geopandas as gpd
import time
import json
import os
import folium
from pyproj import Transformer, CRS
import webbrowser
import pandas as pd
from input_calculations.X_GLOBAL_SCRIPTS import helper_functions as hf
pd.options.mode.chained_assignment = None

start_time = time.time()
################################################################################################################################


################################################################################################################################
print('### DATA IMPORT ###')

# Check if required datasets exists
assert os.path.exists('../X_GLOBAL/data_processed/dataset_fylke.feather'), 'Create dataset_fylke first!'

# Import base CRS
with open('../X_GLOBAL/data_processed/crs.json', 'r') as openfile:
    base_crs = CRS(json.load(openfile))

# Read dataset tettsteder
print('Reading tettsted2024.geojson')
gdf_tettsteder = gpd.read_file('../X_GLOBAL/data_raw/general/tettsted2024.geojson')
gdf_tettsteder.crs = base_crs  # The dataset comes with a wrong CRS

# Read dataset electricity price zones
print('Reading strømprissoner.geojson')
gdf_electricity_zones = gpd.read_file('../X_GLOBAL/data_raw/general/strømprissoner.geojson')
gdf_electricity_zones.to_crs(base_crs, inplace=True)

# Reading fylke geography (multipolygon of each landsdel)
gdf_fylkedata = gpd.read_feather('../X_GLOBAL/data_processed/dataset_fylke.feather')

print('Time passed: {:.2f} seconds'.format(time.time() - start_time))
################################################################################################################################


################################################################################################################################
print('\n### CALCULATION ###')

# Format electricity zone dataframe
gdf_electricity_zones['ElSpotOmr'] =  gdf_electricity_zones['ElSpotOmr'].str.replace(" ", "")
gdf_electricity_zones.index = gdf_electricity_zones['ElSpotOmr']
gdf_electricity_zones = gdf_electricity_zones.loc[:,['geometry']]
hf.plot_selected_geodataframe(gdf_electricity_zones,'electricity_zones', folder='plots/')

# Transform input points from EPSG:25833 to EPSG:4326
transformer_25833_to_4326 = Transformer.from_crs("EPSG:25833", "EPSG:4326")

# Plot tettsteder with population > 5.000
gdf_tettsteder = gdf_tettsteder[gdf_tettsteder['tot_bef'] > 5000]
gdf_tettsteder['centroid_25833'] = gdf_tettsteder.geometry.centroid
gdf_tettsteder['centroid_4326'] = gdf_tettsteder.apply(lambda row: tuple(reversed(transformer_25833_to_4326.transform(row.centroid_25833.x, row.centroid_25833.y))), axis=1)
gdf_tettsteder['geometry'] = gdf_tettsteder['centroid_25833'].copy()
gdf_tettsteder.drop('centroid_25833', axis='columns', inplace=True)

# Add electricity zone to tettsted
gdf_tettsteder = gpd.sjoin(gdf_tettsteder, gdf_electricity_zones, how='left', predicate='within')

# Add landsdel to tettsted
gdf_tettsteder = gpd.sjoin(gdf_tettsteder, gdf_fylkedata[['geometry', 'landsdel']], how='left', predicate='within')
gdf_tettsteder.drop(['fylkesnummer', 'tot_bef', 'tettnr', 'areal_km2', 'beff_tett'], axis='columns', inplace=True)

# Plot population centers
m = folium.Map(tiles="cartodb positron")
_ = gdf_tettsteder.apply(lambda row: folium.Marker(location=tuple(reversed(row.centroid_4326)), icon=folium.Icon(color='black', icon_color='black'), tooltip=row.tettstedsn).add_to(m), axis=1)
plot_name = 'tettsteder_norway'
m.save('plots/' + plot_name + ".html")
webbrowser.open_new_tab(os.getcwd() + '/plots/' + plot_name + ".html")

# Export
gdf_tettsteder.to_feather('../X_GLOBAL/data_processed/dataset_tettsteder.feather')
gdf_electricity_zones.to_feather('../X_GLOBAL/data_processed/dataset_electricity_zones.feather')
################################################################################################################################
