#################################################################################################################################################################################################################################################
# AUTHOR: Matthias Maier
# Task: Calculate the transport distance between points
#################################################################################################################################################################################################################################################


######################################################################################################################################################
# IMPORT
import openrouteservice
import folium
import webbrowser
import numpy as np
from pyproj import Transformer
from shapely import Point
from shapely.ops import transform as shapely_transform
import random
import pandas as pd
import geopandas as gpd
import os
import time
import multiprocessing
from input_calculations.X_GLOBAL_SCRIPTS import network_operations_graph as nx

######################################################################################################################################################


######################################################################################################################################################

def calculate_route_between_two_points(start_point, end_point, plot_name = None, use_local_server = True, transform_input_points = (True, True), try_alternatives=True, gdf_road_network = None, debug=False):
    """Function to calculate distance and driving time between two points

    :param start_point: Start point using EPSG:25833 Point(latitude, longitude) if transform else EPSG:4326 tuple(longitude, latitude)
    :param end_point: End point using EPSG:25833 Point(latitude, longitude) if transform else EPSG:4326 tuple(longitude, latitude)
    :param plot_name: Name of the plot (no file ending). If specified, a folium plot is generated
    :param use_local_server: Use local server (hosted via docker at localhost)
    :param transform_input_points: Transform crs from EPSG:25833 (standard for reduced centers) to EPSG:4326 (used for querying openrouteservice)
    :param try_alternatives: If OpenRouteService cannot find a route, try different start points in the proximity of using Elveg2 and estimate the
                             distance from the original start point and the alternative start point manually
    :param gdf_road_network: GeoDataFrame with road network (necessary and only effective when try_alternatives is set to True)
    """

    ###########################################################################################################
    # SETUP

    if try_alternatives:
        assert gdf_road_network is not None, 'Specify road dataset!'
        assert gdf_road_network.ndim == 2, 'The road dataset must be 2D (x,y). Check for z dimensions!'

    # Set up parameters
    solution_found = False # If ORS found a feasible route
    search_radius = 400 # [m]
    max_search_radius = 5000 # [m]
    routes_ors = None
    route_elv2 = None
    lin_estimation_share = 0 # % of route which is estimated using linear estimates

    # Transform input points from EPSG:25833 to EPSG:4326
    transformer_25833_to_4326 = Transformer.from_crs("EPSG:25833", "EPSG:4326")
    transformer_4326_to_25833 = Transformer.from_crs("EPSG:4326", "EPSG:25833", always_xy=True)

    if transform_input_points[0]: # Transform start point
        start_point_25833 = start_point
        start_point = tuple(reversed(transformer_25833_to_4326.transform(start_point.x, start_point.y)))
    else:
        start_point_25833 = shapely_transform(transformer_4326_to_25833.transform, Point(start_point))

    if transform_input_points[1]: # Transform end point
        end_point_25833 = end_point
        end_point = tuple(reversed(transformer_25833_to_4326.transform(end_point.x, end_point.y)))
    else:
        end_point_25833 = shapely_transform(transformer_4326_to_25833.transform, Point(end_point))

    assert type(start_point) == tuple, 'Start point must be tuple if transform is false!'
    assert type(end_point) == tuple, 'Start point must be tuple if transform is false!'

    # The format of the start and end point will be EPSG:4326 tuple(longitude, latitude) after here
    ###########################################################################################################

    ###########################################################################################################
    # SEARCH FOR A ROUTE WITH ORS
    # If the areal distance between 2 places is less than 5k, they are considered the same place

    areal_distance = end_point_25833.distance(start_point_25833) # [m]

    if areal_distance < 5000:
        distance = 0
        driving_time = 0
        solution_found = True
        print('\tResolution: The two places are less than 5k apart and are considered the same place')
    else:
        # Choose client (server or local)
        if use_local_server:
            client = openrouteservice.Client(base_url='http://localhost:8080/ors')
        else:
            API_key = '5b3ce3597851110001cf6248a522e974a0144580b6f6d1423fb73739'
            client = openrouteservice.Client(key=API_key)

        # Try finding an OSR solution given a search radius. If no solution can be found, increase search radius gradually up to 5000m
        while search_radius < max_search_radius and solution_found == False:
            try:
                coords = (start_point, end_point)  # longitude, latitude
                routes_ors = client.directions(coordinates=coords, profile='driving-car', format='geojson', radiuses = (search_radius, search_radius))

                distance = routes_ors['features'][0]['properties']['summary']['distance'] # Distance of the route [m]
                driving_time = routes_ors['features'][0]['properties']['summary']['duration'] # Driving time [sec]

                solution_found = True
            except Exception as e:
                search_radius = search_radius + 100
    ###########################################################################################################

    ###########################################################################################################
    # ALTERNATIVE ROUTING
    # If OpenRouteService could not find a solution, try a combined approach (Elveg2 + ORS)

    if solution_found is False and try_alternatives is False:
        print('\tORS: No route could be found')

        distance = np.inf
        driving_time = np.inf

    if solution_found is False and try_alternatives is True:
        print('\tNo route could be found using OpenRouteService, but alternative start points in the close proximity are tried')
        distance, driving_time, search_radius, routes_ors, route_elv2, lin_estimation_share = _look_for_alternative_routes(start_point, end_point, gdf_road_network.copy(deep=True), debug=debug)
    ###########################################################################################################

    ###########################################################################################################
    # OUTPUT & PLOTTING

    if search_radius > 400 and solution_found:
        print('\tORS: Route could be found, but search radius was increased to ' + str(search_radius) + ' meters')

    if plot_name is not None:
        m = folium.Map(location=(start_point[1], start_point[0]), tiles="cartodb positron")

        if routes_ors is not None:
            folium.PolyLine(locations=[list(reversed(coord)) for coord in routes_ors['features'][0]['geometry']['coordinates']], color='blue', tooltip='Route from OpenRouteService').add_to(m)

        if route_elv2 is not None:
            route_elv2_as_points_25833 = [Point(el) for el in route_elv2.coords]
            route_elv2_as_coords_4326 = [tuple(transformer_25833_to_4326.transform(el.x, el.y)) for el in route_elv2_as_points_25833]

            folium.PolyLine(locations=route_elv2_as_coords_4326, color='black', tooltip='Route from Elveg2 (Info: Route is approximate, but the distance value is precise!)').add_to(m)

        folium.Marker(location=tuple(reversed(start_point)), icon=folium.Icon(color='black', icon_color='black'), tooltip='Start').add_to(m)
        folium.Marker(location=tuple(reversed(end_point)), icon=folium.Icon(color='black', icon_color='black'), tooltip='Destination').add_to(m)

        m.save('outputs/' + plot_name + ".html")
        webbrowser.open_new_tab(os.getcwd() + '/outputs/' + plot_name + ".html")

    return distance, driving_time, search_radius, lin_estimation_share, routes_ors
    ###########################################################################################################

######################################################################################################################################################

######################################################################################################################################################
def _look_for_alternative_routes(original_start_point, end_point, gdf_road_network, debug=False):
    """
    :param original_start_point: Given as a tuple in CRS 4326 (Original point, where ORS could not find a route from)
    :param end_point: Given as a tuple in CRS 4326 (End point, where ORS could not find a route to)
    :param gdf_road_network: The road network geodataframe
    :param debug: If true, create a folium plot for the pairs where Elveg2 could not find a route or ORS could not find a route
    :return: distance, driving_time, search_radius, routes_ors, route_elv2
    """

    # Set up transformers
    transformer_4326_to_25833 = Transformer.from_crs("EPSG:4326", "EPSG:25833", always_xy=True)
    transformer_25833_to_4326 = Transformer.from_crs("EPSG:25833", "EPSG:4326")

    # Original start point
    original_start_point_25833 = shapely_transform(transformer_4326_to_25833.transform, Point(original_start_point))

    assert min(gdf_road_network.distance(original_start_point_25833)) < 100  # Should be 0 since the start point is part of the road dataset

    ###########################################################################################################
    # CREATE LIST OF ALTERNATIVE START POINTS
    # 1. Look for bigger roads in the closer proximity to the original start point (speed limit >= 80)
    # 2. Select up to 40 random road segments
    #    a) Look for up to 20 points close to the original start point (unlikely for ORS but likely for Elveg2)
    #    b) Look for up to 20 points further away from the original start point (likely for ORS but difficult for Elveg2))
    # 3. Use the start and end point of the obtained road segments as the base list for new starting points for ORS

    # Initialize values
    gdf_road_network_80 = gdf_road_network[gdf_road_network['fartsgrenseVerdi'] >= 80]
    road_distance_to_original_startpoint = gdf_road_network_80.distance(original_start_point_25833)
    search_point_list = [] # List of search points, ascending according to distance to orginal starting point
    dist_closest_road = np.inf
    dist_furthest_road = 0

    for distance_limit in [(0,20000), (20000,50000)]: # Distance to look for roads in the area around the original start point [m]

        gdf_road_network_i = gdf_road_network_80[(road_distance_to_original_startpoint >= distance_limit[0]) & (road_distance_to_original_startpoint <= distance_limit[1])]

        # Extract the start point and the end point of all road segments
        gdf_road_network_i = pd.concat([gdf_road_network_i, gdf_road_network_i[['geometry']].map(lambda el: Point(el.coords[0])).rename(columns={'geometry': 'road_start_point'})], axis='columns')
        gdf_road_network_i = pd.concat([gdf_road_network_i, gdf_road_network_i[['geometry']].map(lambda el: Point(el.coords[-1])).rename(columns={'geometry': 'road_end_point'})], axis='columns')

        # Select up to 20 points from the search point list
        search_point_list_i = list(set(gdf_road_network_i['road_start_point'].values).union(set(gdf_road_network_i['road_end_point'].values)))
        search_point_list_i = random.sample(search_point_list_i, min(20, len(search_point_list_i)))

        # Sort the search point list according to distance to original start point (closer points are more likely to be feasible with Elveg2, so start with them)
        gdf_search_point_list = gpd.GeoDataFrame(geometry=search_point_list_i, crs=gdf_road_network.crs)
        gdf_search_point_list['distance'] = gdf_search_point_list.geometry.distance(original_start_point_25833)
        gdf_search_point_list.sort_values('distance', inplace=True)
        search_point_list_i = gdf_search_point_list['geometry'].tolist()

        search_point_list += search_point_list_i

        if gdf_search_point_list['distance'].min() / 1000 < dist_closest_road:
            dist_closest_road = gdf_search_point_list['distance'].min() / 1000

        if gdf_search_point_list['distance'].max() / 1000 > dist_furthest_road:
            dist_furthest_road = gdf_search_point_list['distance'].max() / 1000

    # If no search point could be found, exit
    if len(search_point_list) == 0:
        print('\tNo road with speed limit could be found in a 50km radius. Search discontinued')

        distance = np.inf
        driving_time = np.inf

        return distance, driving_time, -1, None, None, -1 # distance, driving_time, search_radius, routes_ors, route_elv2, lin_estimation_share

    print('\tA list of ' + str(len(search_point_list)) + ' alternative start points found ({:.2f} - {:.2f} km away). Trying to find a feasible OpenRouteService with one of these points\n'.format(dist_closest_road, dist_furthest_road))
    print('\t### TRYING ALTERNATIVE POINTS - START ###')
    ###########################################################################################################

    ###########################################################################################################
    # TRY ALTERNATIVE START POINTS

    for alternative_start_point_25833 in search_point_list:

        # Format alternative start point
        alternative_start_point_4326_shapely = shapely_transform(transformer_25833_to_4326.transform, alternative_start_point_25833)
        alternative_start_point_4326 = (alternative_start_point_4326_shapely.y, alternative_start_point_4326_shapely.x)

        # Try OpenRouteService for the alternative start point
        distance, driving_time, search_radius, _, routes_ors = calculate_route_between_two_points(alternative_start_point_4326, end_point, transform_input_points=(False, False), try_alternatives=False)

        # If it worked, estimate distance from the original starting point to the alternative starting point
        if distance < np.inf:

            print('\tORS: Route could be found using alternative point!')
            print('\tNow trying to estimate distance and driving time from the original starting point to the alternative starting point using Elveg2\n')

            # Straight line distance
            areal_distance_altSP_to_orgSP = alternative_start_point_25833.distance(original_start_point_25833)

            # Slice the dataset to get the road segments which are roughly between the original start point and the alternative start point
            dataset = gdf_road_network[gdf_road_network.distance(original_start_point_25833) <= areal_distance_altSP_to_orgSP * 1.2]

            # Both the original start point and the alternative start point might not be exactly in the road dataset due to comma inaccuracies
            # Thus, both points need to be translated into points which are in the road dataset

            original_start_point_25833_in_dataset = _translate_point_to_point_in_dataset(original_start_point_25833, dataset.copy(deep=True))
            alternative_start_point_25833_in_dataset = _translate_point_to_point_in_dataset(alternative_start_point_25833, dataset.copy(deep=True))

            # Try finding an Elveg2 route. If no route found after 30s, timeout
            time_before_Elveg2 = time.time()
            distance_to_start_point, route_elv2, timeout_occured = _calculate_Elveg2_route_in_seperate_process(original_start_point_25833_in_dataset, alternative_start_point_25833_in_dataset, dataset, 30)

            print('\t\tThis took {:.2f} seconds\n'.format(time.time() - time_before_Elveg2))

            if distance_to_start_point < np.inf:
                print('\tRoute from the original starting point to the alternative starting point could be found using Elveg2!')
                print('\tAppending distance and time from the original starting point to the alternative starting point to the results from OpenRouteService')

            else:
                print('\tNo route from the original starting point to the alternative starting point could be found using Elveg2')
                print('\tEstimating distance and driving time from the original starting point to the alternative starting point using straight line estimates')

                distance_to_start_point = original_start_point_25833.distance(alternative_start_point_25833) * 1.5

                if debug:
                    m = folium.Map(location=(original_start_point[1], original_start_point[0]), tiles="cartodb positron")
                    _add_point_4326_to_map(m, original_start_point, tooltip='Original start point')
                    _add_point_4326_to_map(m, alternative_start_point_4326, tooltip='Alternative start point (ORS found a route from here)')
                    _add_point_4326_to_map(m, end_point, tooltip='End point')

                    if len(dataset.geometry.to_list()) < 10000:
                        _add_linestrings_25833_to_map(m, dataset.geometry.to_list())

                    if timeout_occured:
                        filename = 'debug_plot__timeoute__ORS_found_route_from_altPT_but_Elveg2_did_not__' + str(original_start_point) + ' to ' + str(end_point) + '.html'
                    else:
                        filename = 'debug_plot__ORS_found_route_from_altPT_but_Elveg2_did_not__' + str(original_start_point) + ' to ' + str(end_point) + '.html'

                    _save_map_to_debug_folder(m, filename)

            distance = distance + distance_to_start_point
            driving_time = driving_time + distance_to_start_point / 1000 / 50 * 3600 # Assuming 50 km/h on the path from the original start point to the alternative start point

            print('\tShare of distance (original starting point - alternative starting point) / total distance: {:.2f}'.format(distance_to_start_point/distance))

            lin_estimation_share = distance_to_start_point / distance * 100

            return distance, driving_time, search_radius, routes_ors, route_elv2, lin_estimation_share # distance, driving_time, search_radius, routes_ors, route_elv2, lin_estimation_share

    print('\t### TRYING ALTERNATIVE POINTS - END ###\n')
    print('\tNo feasible path using OpenRouteService and the alternative start points could be found. Infeasible journey')

    if debug:
        m = folium.Map(location=(original_start_point[1], original_start_point[0]), tiles="cartodb positron")
        _add_point_4326_to_map(m, original_start_point, tooltip='Original start point')
        _add_point_4326_to_map(m, end_point, tooltip='End point')
        _add_linestrings_25833_to_map(m, gdf_road_network.geometry.to_list())

        _save_map_to_debug_folder(m, 'debug_plot__ORS_did_not_find_a_route__' + str(original_start_point) + ' to ' + str(end_point) + '.html')

    return np.inf, np.inf, -1, None, None,-1 # distance, driving_time, search_radius, routes_ors, route_elv2, lin_estimation_share

    ###########################################################################################################

######################################################################################################################################################


######################################################################################################################################################
def _translate_point_to_point_in_dataset(point, dataset):

    # Distance of the road segments to the point
    dataset['distance'] = dataset.distance(point)
    dataset = dataset.sort_values('distance')
    assert min(dataset['distance']) < 100

    # Get the closest road and extract Point(coords) into dataframe
    closest_road_extracted = gpd.GeoDataFrame(geometry=[Point(el) for el in dataset.iloc[0, :]['geometry'].coords], crs=dataset.crs)

    # Sort road points according to distance to point and get the closest road point
    closest_road_extracted['distance'] = closest_road_extracted.distance(point)
    closest_road_extracted = closest_road_extracted.sort_values('distance')
    point = closest_road_extracted.iloc[0, :]['geometry']

    return point
######################################################################################################################################################


######################################################################################################################################################
def _calculate_Elveg2_route_in_seperate_process(original_start_point_25833_in_dataset, alternative_start_point_25833_in_dataset, dataset, timeout):

    print('\t\tStarting seperate process for Elveg2 with timeout of ' + str(timeout) + ' seconds')

    timeout_occured = False
    manager = multiprocessing.Manager()
    return_dict = manager.dict()

    process = multiprocessing.Process(target=nx.calculate_Elveg2_route, args=[return_dict, original_start_point_25833_in_dataset, alternative_start_point_25833_in_dataset, dataset])
    process.start()
    process.join(timeout) # Wait until process terminates or until timeout

    if process.is_alive():
        print('\t\tTimeout: Terminating process forcefully')
        timeout_occured = True
        process.terminate()
        time.sleep(1)

    # Process should be dead after process.terminate() but it might take some time
    if process.is_alive():
        time.sleep(5)
        assert not process.is_alive()

    if timeout_occured:
        distance = np.inf
        route = None
    else:
        distance = return_dict['distance']
        route = return_dict['route']

    print('\t\tSeparate process finished and process terminated')
    return distance, route, timeout_occured
######################################################################################################################################################

######################################################################################################################################################
# DEBUG FUNCTIONS

def _add_linestrings_25833_to_map(map, linestringlist_25833):

    transformer_25833_to_4326 = Transformer.from_crs("EPSG:25833", "EPSG:4326")

    for linstring in linestringlist_25833:
        route_as_points = [Point(el) for el in linstring.coords]
        route_elv2_as_coords_4326 = [tuple(transformer_25833_to_4326.transform(el.x, el.y)) for el in route_as_points]

        folium.PolyLine(locations=route_elv2_as_coords_4326, color='black').add_to(map)

def _add_point_4326_to_map(map, point, tooltip=''):
    folium.Marker(location=tuple(reversed(point)), icon=folium.Icon(color='black', icon_color='black'), tooltip=tooltip).add_to(map)

def _save_map_to_debug_folder(map, plot_name):

    cwd = os.getcwd()
    debug_folder = os.path.join(cwd, "debug_plots")

    # Create the folder if it doesn't exist
    os.makedirs(debug_folder, exist_ok=True)

    file_path = os.path.join(debug_folder, plot_name)
    map.save(file_path)

def _show_map(map):
    map.save('map_debug.html')
    webbrowser.open_new_tab("map_debug.html")

######################################################################################################################################################